<?php
function scanDirectories($dir) {
    $dirs = [];
    if (!file_exists($dir)) return $dirs;
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        $path = $dir . '/' . $item;
        if (is_dir($path)) $dirs[$item] = $path;
    }
    return $dirs;
}
$mediaDir = __DIR__ . '/assets';
$directories = scanDirectories($mediaDir);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Imators Gallery</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600&display=swap" rel="stylesheet">
    <script>
        tailwind.config = {
            darkMode: 'media',
            theme: {
                extend: {
                    fontFamily: { sans: ['Inter', 'sans-serif'] },
                    aspectRatio: { portrait: '3 / 4' }
                }
            }
        }
    </script>
    <style>
        body { font-family: 'Inter', sans-serif; }
        .modal { 
            display: none; 
            position: fixed; 
            top: 0; 
            left: 0; 
            width: 100%; 
            height: 100vh; 
            background: rgba(0, 0, 0, 0.95); 
            z-index: 1000; 
            justify-content: center; 
            align-items: center;
            backdrop-filter: blur(20px);
        }
        .modal-content { 
            max-width: 95vw; 
            max-height: 90vh; 
            position: relative;
        }
        .modal img, .modal video { 
            max-height: 85vh; 
            max-width: 90vw; 
            object-fit: contain; 
            border-radius: 4px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
        }
        .sidebar { 
            transform: translateX(-100%); 
            transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        .sidebar.open { 
            transform: translateX(0); 
        }
        .overlay { 
            display: none; 
            position: fixed; 
            top: 0; 
            left: 0; 
            width: 100%; 
            height: 100%; 
            background: rgba(0, 0, 0, 0.4); 
            z-index: 40;
        }
        .overlay.show { 
            display: block; 
        }
        .loading { 
            display: none; 
            justify-content: center; 
            align-items: center; 
            padding: 60px;
        }
        .spinner { 
            width: 24px; 
            height: 24px; 
            border: 2px solid; 
            border-color: transparent;
            border-top-color: currentColor;
            border-radius: 50%; 
            animation: spin 1s linear infinite;
        }
        @keyframes spin { 
            0% { transform: rotate(0deg); } 
            100% { transform: rotate(360deg); } 
        }
        .media-item { 
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        .media-item:hover { 
            transform: translateY(-1px);
        }
        .welcome-screen {
            transition: opacity 0.5s ease-out;
        }
        .welcome-screen.fade-out {
            opacity: 0;
            pointer-events: none;
        }
        @media (min-width: 768px) {
            .sidebar {
                transform: translateX(0);
                position: relative;
            }
        }
        @media (max-width: 768px) {
            .modal img, .modal video { 
                max-height: 80vh; 
                max-width: 95vw; 
            }
        }
    </style>
</head>
<body class="bg-white dark:bg-black text-black dark:text-white">
    <div id="welcome-screen" class="welcome-screen fixed inset-0 z-50 flex items-center justify-center bg-white dark:bg-black">
        <div class="text-center px-8">
            <div class="mb-8">
                <div class="w-16 h-16 mx-auto mb-6 border border-black/10 dark:border-white/10 rounded-lg flex items-center justify-center">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                    </svg>
                </div>
                <h1 class="text-3xl font-light mb-3">Welcome in InCloud</h1>
                <p class="text-sm text-black/60 dark:text-white/60">Cloud created and developed by Izhak Aktas</p>
            </div>
            <button onclick="hideWelcomeScreen()" class="px-6 py-2 border border-black/20 dark:border-white/20 rounded hover:bg-black/5 dark:hover:bg-white/5 transition-all duration-200 text-sm">
                Enter
            </button>
        </div>
    </div>

    <div class="overlay" id="overlay" onclick="closeSidebar()"></div>
    
    <div class="flex h-screen">
        <div class="sidebar fixed md:relative w-64 bg-white dark:bg-black border-r border-black/10 dark:border-white/10 z-50 h-full md:flex">
            <div class="p-6 w-full">
                <div class="flex items-center justify-between mb-8">
                    <h2 class="text-sm font-medium">Folders</h2>
                    <button onclick="closeSidebar()" class="md:hidden">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                
                <div class="space-y-1 mb-8">
                    <button onclick="loadFolder(''); closeSidebar();" class="folder-btn w-full flex items-center space-x-3 py-2 px-3 rounded hover:bg-black/5 dark:hover:bg-white/5 transition-colors text-left text-sm">
                        <svg class="w-4 h-4 opacity-60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"/>
                        </svg>
                        <span>All Media</span>
                    </button>
                    <?php foreach ($directories as $dirName => $dirPath): ?>
                        <button onclick="loadFolder('<?= htmlspecialchars($dirName) ?>'); closeSidebar();" class="folder-btn w-full flex items-center space-x-3 py-2 px-3 rounded hover:bg-black/5 dark:hover:bg-white/5 transition-colors text-left text-sm">
                            <svg class="w-4 h-4 opacity-60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                            </svg>
                            <span><?= htmlspecialchars($dirName) ?></span>
                        </button>
                    <?php endforeach; ?>
                </div>
                
                <div class="border-t border-black/10 dark:border-white/10 pt-6">
                    <h3 class="text-xs font-medium mb-4 opacity-60 uppercase tracking-wider">Filters</h3>
                    <div class="space-y-3">
                        <label class="flex items-center space-x-3 cursor-pointer">
                            <input type="checkbox" class="filter-checkbox w-3 h-3 border border-black/20 dark:border-white/20 rounded-sm" data-type="image" checked>
                            <div class="flex items-center space-x-2">
                                <svg class="w-4 h-4 opacity-60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                </svg>
                                <span class="text-sm">Images</span>
                            </div>
                        </label>
                        <label class="flex items-center space-x-3 cursor-pointer">
                            <input type="checkbox" class="filter-checkbox w-3 h-3 border border-black/20 dark:border-white/20 rounded-sm" data-type="video" checked>
                            <div class="flex items-center space-x-2">
                                <svg class="w-4 h-4 opacity-60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z"/>
                                </svg>
                                <span class="text-sm">Videos</span>
                            </div>
                        </label>
                        <label class="flex items-center space-x-3 cursor-pointer">
                            <input type="checkbox" class="filter-checkbox w-3 h-3 border border-black/20 dark:border-white/20 rounded-sm" data-type="audio" checked>
                            <div class="flex items-center space-x-2">
                                <svg class="w-4 h-4 opacity-60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3"/>
                                </svg>
                                <span class="text-sm">Audio</span>
                            </div>
                        </label>
                    </div>
                </div>
            </div>
        </div>

        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="border-b border-black/10 dark:border-white/10 p-6">
                <div class="flex items-center justify-between">
                    <div class="flex items-center space-x-4">
                        <button onclick="openSidebar()" class="md:hidden">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                            </svg>
                        </button>
                        <div>
                            <h1 class="text-xl font-medium">Gallery</h1>
                            <p class="text-xs opacity-60 mt-1">InCloud, an Imators Products, all right reserved.</p>
                        </div>
                    </div>
                    <div class="text-xs opacity-60" id="media-count">Loading...</div>
                </div>
            </header>

            <main class="flex-1 p-6 overflow-y-auto">
                <div class="loading" id="loading">
                    <div class="text-center">
                        <div class="spinner mx-auto mb-4 text-black/40 dark:text-white/40"></div>
                        <p class="text-xs opacity-60">Loading...</p>
                    </div>
                </div>
                <div id="media-grid" class="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-4 xl:grid-cols-6 gap-3"></div>
                <div id="empty-state" class="hidden text-center py-20">
                    <div class="w-12 h-12 mx-auto mb-4 border border-black/10 dark:border-white/10 rounded flex items-center justify-center">
                        <svg class="w-6 h-6 opacity-40" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                        </svg>
                    </div>
                    <h3 class="text-sm font-medium mb-2">No media found</h3>
                    <p class="text-xs opacity-60">This folder is empty</p>
                </div>
            </main>
        </div>
    </div>

    <div id="media-modal" class="modal">
        <div class="modal-content">
            <button id="close-modal" class="absolute -top-8 right-0 text-white/80 hover:text-white transition-colors z-10">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
            <img id="modal-image" class="hidden">
            <video id="modal-video" class="hidden" controls preload="metadata">
                <source id="modal-video-source" type="">
            </video>
            <audio id="modal-audio" class="hidden w-full rounded p-4" controls style="background: #1a1a1a;">
                <source id="modal-audio-source" type="">
            </audio>
        </div>
    </div>

    <div class="fixed bottom-0 left-0 right-0 bg-white/95 dark:bg-black/95 backdrop-blur-sm border-t border-black/10 dark:border-white/10 p-4 z-30">
        <div class="max-w-6xl mx-auto flex items-center justify-between text-xs">
            <div class="flex items-center space-x-4 opacity-70">
                <div class="flex items-center space-x-2">
                    <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.031 9-11.622 0-1.042-.133-2.052-.382-3.016z"/>
                    </svg>
                    <span>Content secured and preserved</span>
                </div>
                <div class="hidden sm:block">•</div>
                <div class="hidden sm:block">Archive dates: 2004-2017</div>
                <div class="hidden sm:block">•</div>
                <div class="hidden sm:block">Files may be unstable</div>
            </div>
            <div class="text-right">
                <span class="opacity-60">Download recommended for preservation</span>
            </div>
        </div>
    </div>

    <script>
        let currentFolder = '';
        let allMediaItems = [];

        function hideWelcomeScreen() {
            const welcomeScreen = document.getElementById('welcome-screen');
            welcomeScreen.classList.add('fade-out');
            setTimeout(() => {
                welcomeScreen.remove();
                loadFolder('');
            }, 500);
        }

        function openSidebar() {
            document.querySelector('.sidebar').classList.add('open');
            document.getElementById('overlay').classList.add('show');
        }

        function closeSidebar() {
            document.querySelector('.sidebar').classList.remove('open');
            document.getElementById('overlay').classList.remove('show');
        }

        function openModal(src, type, name) {
            const modal = document.getElementById('media-modal');
            const image = document.getElementById('modal-image');
            const video = document.getElementById('modal-video');
            const videoSource = document.getElementById('modal-video-source');
            const audio = document.getElementById('modal-audio');
            const audioSource = document.getElementById('modal-audio-source');

            image.classList.add('hidden');
            video.classList.add('hidden');
            audio.classList.add('hidden');

            if (type === 'image') {
                image.src = src;
                image.alt = name;
                image.classList.remove('hidden');
            } else if (type === 'video') {
                videoSource.src = src;
                videoSource.type = getVideoMimeType(src);
                video.load();
                video.classList.remove('hidden');
            } else if (type === 'audio') {
                audioSource.src = src;
                audioSource.type = getAudioMimeType(src);
                audio.load();
                audio.classList.remove('hidden');
            }

            modal.style.display = 'flex';
            document.body.style.overflow = 'hidden';
        }

        function closeModal() {
            const modal = document.getElementById('media-modal');
            const video = document.getElementById('modal-video');
            const audio = document.getElementById('modal-audio');
            
            video.pause();
            audio.pause();
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }

        function getVideoMimeType(src) {
            const ext = src.split('.').pop().toLowerCase();
            const types = {
                'mp4': 'video/mp4',
                'mov': 'video/quicktime',
                'avi': 'video/x-msvideo',
                'webm': 'video/webm'
            };
            return types[ext] || 'video/mp4';
        }

        function getAudioMimeType(src) {
            const ext = src.split('.').pop().toLowerCase();
            const types = {
                'mp3': 'audio/mpeg',
                'wav': 'audio/wav',
                'ogg': 'audio/ogg',
                'aac': 'audio/aac'
            };
            return types[ext] || 'audio/mpeg';
        }

        function loadFolder(folderName) {
            currentFolder = folderName;
            document.getElementById('loading').style.display = 'flex';
            document.getElementById('media-grid').innerHTML = '';
            document.getElementById('empty-state').classList.add('hidden');

            const xhr = new XMLHttpRequest();
            xhr.open('GET', 'get_folder.php?folder=' + encodeURIComponent(folderName), true);
            xhr.onload = function() {
                document.getElementById('loading').style.display = 'none';
                if (xhr.status === 200) {
                    const response = xhr.responseText.trim();
                    if (response) {
                        document.getElementById('media-grid').innerHTML = response;
                        allMediaItems = document.querySelectorAll('.media-item');
                        updateMediaCount();
                        applyFilters();
                    } else {
                        document.getElementById('empty-state').classList.remove('hidden');
                        updateMediaCount(0);
                    }
                } else {
                    document.getElementById('media-grid').innerHTML = '<div class="col-span-full text-center py-8 text-red-500 text-sm">Error loading media</div>';
                }
            };
            xhr.onerror = function() {
                document.getElementById('loading').style.display = 'none';
                document.getElementById('media-grid').innerHTML = '<div class="col-span-full text-center py-8 text-red-500 text-sm">Connection error</div>';
            };
            xhr.send();
        }

        function updateMediaCount(count = null) {
            if (count === null) {
                const visibleItems = document.querySelectorAll('.media-item:not([style*="display: none"])');
                count = visibleItems.length;
            }
            document.getElementById('media-count').textContent = `${count} file${count !== 1 ? 's' : ''}`;
        }

        function applyFilters() {
            const imageChecked = document.querySelector('.filter-checkbox[data-type="image"]').checked;
            const videoChecked = document.querySelector('.filter-checkbox[data-type="video"]').checked;
            const audioChecked = document.querySelector('.filter-checkbox[data-type="audio"]').checked;

            allMediaItems.forEach(item => {
                const type = item.dataset.type;
                const shouldShow = (type === 'image' && imageChecked) || 
                                 (type === 'video' && videoChecked) || 
                                 (type === 'audio' && audioChecked);
                item.style.display = shouldShow ? 'block' : 'none';
            });

            updateMediaCount();
        }

        document.getElementById('close-modal').addEventListener('click', closeModal);

        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
                closeSidebar();
            }
        });

        document.getElementById('media-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        document.querySelectorAll('.filter-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', applyFilters);
        });

        function dismissNotice() {
            document.querySelector('.fixed.bottom-0').style.display = 'none';
        }
    </script>
</body>
</html>