<?php
$current_file = basename($_SERVER['PHP_SELF']);

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' B';
    }
}

function getAllFiles($directory = '.') {
    global $current_file;
    $files = [];
    $items = scandir($directory);
    
    foreach ($items as $item) {
        if ($item === '.' || $item === '..' || $item === $current_file) continue;
        
        $path = $directory . DIRECTORY_SEPARATOR . $item;
        
        if (is_file($path)) {
            $files[] = [
                'name' => $item,
                'path' => $path,
                'size' => filesize($path),
                'modified' => filemtime($path),
                'type' => is_file($path) ? 'file' : 'directory'
            ];
        }
    }
    
    return $files;
}

if (isset($_GET['download']) && !empty($_GET['download'])) {
    $file = $_GET['download'];
    $filepath = './' . $file;
    
    if (file_exists($filepath) && is_file($filepath) && $file !== $current_file) {
        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . basename($file) . '"');
        header('Content-Length: ' . filesize($filepath));
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Expires: 0');
        
        readfile($filepath);
        exit;
    }
}

if (isset($_POST['download_all'])) {
    $files = getAllFiles();
    
    if (!empty($files)) {
        $zip = new ZipArchive();
        $zip_name = 'directory_files_' . date('Y-m-d_H-i-s') . '.zip';
        
        if ($zip->open($zip_name, ZipArchive::CREATE) === TRUE) {
            foreach ($files as $file) {
                if (is_file($file['path'])) {
                    $zip->addFile($file['path'], $file['name']);
                }
            }
            $zip->close();
            
            header('Content-Type: application/zip');
            header('Content-Disposition: attachment; filename="' . $zip_name . '"');
            header('Content-Length: ' . filesize($zip_name));
            
            readfile($zip_name);
            unlink($zip_name);
            exit;
        }
    }
}

$files = getAllFiles();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Downloader - Imators</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: #0a0a0a;
            color: #ffffff;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
            line-height: 1.6;
            min-height: 100vh;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 40px 20px;
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
            padding-bottom: 30px;
            border-bottom: 1px solid #1a1a1a;
        }

        .logo {
            width: 60px;
            height: 60px;
            margin: 0 auto 20px;
            display: block;
            border-radius: 8px;
        }

        .title {
            font-size: 32px;
            font-weight: 500;
            margin-bottom: 10px;
            letter-spacing: -0.5px;
        }

        .subtitle {
            font-size: 16px;
            color: #a0a0a0;
            font-weight: 400;
        }

        .stats {
            display: flex;
            justify-content: center;
            gap: 30px;
            margin: 30px 0;
            flex-wrap: wrap;
        }

        .stat {
            text-align: center;
            padding: 20px;
            background: #111111;
            border: 1px solid #1a1a1a;
            border-radius: 12px;
            min-width: 120px;
        }

        .stat-number {
            font-size: 24px;
            font-weight: 600;
            color: #ffffff;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 13px;
            color: #a0a0a0;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .actions {
            text-align: center;
            margin-bottom: 40px;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            font-family: inherit;
            transition: all 0.2s ease;
            text-decoration: none;
            display: inline-block;
        }

        .btn-primary {
            background: #ffffff;
            color: #000000;
        }

        .btn-primary:hover {
            background: #f0f0f0;
            transform: translateY(-1px);
        }

        .files-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }

        .file-card {
            background: #111111;
            border: 1px solid #1a1a1a;
            border-radius: 12px;
            padding: 20px;
            transition: all 0.2s ease;
        }

        .file-card:hover {
            border-color: #2a2a2a;
            transform: translateY(-2px);
        }

        .file-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
        }

        .file-name {
            font-size: 16px;
            font-weight: 500;
            color: #ffffff;
            word-break: break-all;
        }

        .file-extension {
            background: #1a1a1a;
            color: #a0a0a0;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 11px;
            text-transform: uppercase;
            font-weight: 600;
            letter-spacing: 0.5px;
        }

        .file-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .file-size {
            color: #a0a0a0;
            font-size: 13px;
        }

        .file-date {
            color: #a0a0a0;
            font-size: 13px;
        }

        .file-actions {
            text-align: right;
        }

        .btn-download {
            background: transparent;
            color: #ffffff;
            border: 1px solid #2a2a2a;
            padding: 8px 16px;
            font-size: 12px;
        }

        .btn-download:hover {
            background: #1a1a1a;
            border-color: #3a3a3a;
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #a0a0a0;
        }

        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: #ffffff;
        }

        @media (max-width: 768px) {
            .stats {
                gap: 15px;
            }
            
            .stat {
                min-width: 100px;
                padding: 15px;
            }
            
            .files-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <img src="https://cdn.imators.com/logo.png" alt="Imators Logo" class="logo">
            <h1 class="title">Thanks to be a client ! There is your project.</h1>
            <p class="subtitle">Download files from the current directory</p>
        </div>

        <div class="stats">
            <div class="stat">
                <div class="stat-number"><?php echo count($files); ?></div>
                <div class="stat-label">Files</div>
            </div>
            <div class="stat">
                <div class="stat-number"><?php 
                    $total_size = array_sum(array_column($files, 'size'));
                    echo formatFileSize($total_size);
                ?></div>
                <div class="stat-label">Total Size</div>
            </div>
        </div>

        <?php if (!empty($files)): ?>
            <div class="actions">
                <form method="post" style="display: inline;">
                    <button type="submit" name="download_all" class="btn btn-primary">Download All Files (.zip)</button>
                </form>
            </div>

            <div class="files-grid">
                <?php foreach ($files as $file): ?>
                    <div class="file-card">
                        <div class="file-header">
                            <span class="file-name"><?php echo htmlspecialchars($file['name']); ?></span>
                            <span class="file-extension"><?php echo strtoupper(pathinfo($file['name'], PATHINFO_EXTENSION) ?: 'file'); ?></span>
                        </div>
                        <div class="file-info">
                            <span class="file-size"><?php echo formatFileSize($file['size']); ?></span>
                            <span class="file-date"><?php echo date('M j, Y', $file['modified']); ?></span>
                        </div>
                        <div class="file-actions">
                            <a href="?download=<?php echo urlencode($file['name']); ?>" class="btn btn-download">Download</a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <h3>No Files Found</h3>
                <p>The current directory contains no downloadable files.</p>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>