<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);

// Vérifier le niveau minimal requis
if ($user['level'] < 10) {
    header('Location: index.php?error=' . urlencode('Tu dois être au moins niveau 10 pour accéder aux unions.'));
    exit;
}

$error = null;
$success = null;

// Récupérer l'union de l'utilisateur
$user_union = db_get_one(
    "SELECT u.*, um.role, c.name as country_name
     FROM unions u
     JOIN union_members um ON u.id = um.union_id
     JOIN countries c ON u.country_id = c.id
     WHERE um.user_id = ?",
    [$_SESSION['user_id']]
);

$is_leader = $user_union && $user_union['leader_id'] == $_SESSION['user_id'];

// Traitement des formulaires
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Création d'une nouvelle union
    if (isset($_POST['action']) && $_POST['action'] === 'create_union') {
        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        
        if (empty($name)) {
            $error = "Le nom de l'union est requis.";
        } elseif (strlen($name) < 3 || strlen($name) > 50) {
            $error = "Le nom de l'union doit contenir entre 3 et 50 caractères.";
        } elseif ($user_union) {
            $error = "Tu es déjà membre d'une union.";
        } else {
            // Vérifier si le nom est déjà utilisé
            $existing = db_get_one("SELECT id FROM unions WHERE name = ?", [$name]);
            
            if ($existing) {
                $error = "Ce nom d'union est déjà utilisé.";
            } else {
                $union_data = [
                    'name' => $name,
                    'country_id' => $user['country_id'],
                    'leader_id' => $_SESSION['user_id'],
                    'description' => $description,
                    'level' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                ];
                
                $union_id = db_insert('unions', $union_data);
                
                if ($union_id) {
                    // Ajouter le créateur comme membre
                    $member_data = [
                        'union_id' => $union_id,
                        'user_id' => $_SESSION['user_id'],
                        'role' => 'leader',
                        'joined_at' => date('Y-m-d H:i:s')
                    ];
                    
                    db_insert('union_members', $member_data);
                    
                    $success = "Union créée avec succès!";
                    
                    // Recharger les données de l'union
                    $user_union = db_get_one(
                        "SELECT u.*, um.role, c.name as country_name
                         FROM unions u
                         JOIN union_members um ON u.id = um.union_id
                         JOIN countries c ON u.country_id = c.id
                         WHERE um.user_id = ?",
                        [$_SESSION['user_id']]
                    );
                    
                    $is_leader = $user_union['leader_id'] == $_SESSION['user_id'];
                } else {
                    $error = "Erreur lors de la création de l'union.";
                }
            }
        }
    }
    // Quitter une union
    elseif (isset($_POST['action']) && $_POST['action'] === 'leave_union') {
        if (!$user_union) {
            $error = "Tu n'es pas membre d'une union.";
        } elseif ($is_leader) {
            $error = "Le leader ne peut pas quitter l'union. Transfère d'abord le leadership à un autre membre.";
        } else {
            $deleted = db_delete('union_members', 'union_id = ? AND user_id = ?', [$user_union['id'], $_SESSION['user_id']]);
            
            if ($deleted) {
                $success = "Tu as quitté l'union.";
                $user_union = null;
                $is_leader = false;
            } else {
                $error = "Erreur lors de la tentative de quitter l'union.";
            }
        }
    }
    // Inviter un membre
    elseif (isset($_POST['action']) && $_POST['action'] === 'invite_member') {
        $username = trim($_POST['username'] ?? '');
        
        if (empty($username)) {
            $error = "Le nom d'utilisateur est requis.";
        } elseif (!$user_union) {
            $error = "Tu n'es pas membre d'une union.";
        } elseif (!$is_leader && $user_union['role'] != 'officer') {
            $error = "Seuls les leaders et officiers peuvent inviter des membres.";
        } else {
            $target_user = db_get_one("SELECT id, level FROM users WHERE username = ?", [$username]);
            
            if (!$target_user) {
                $error = "Utilisateur non trouvé.";
            } elseif ($target_user['level'] < 10) {
                $error = "L'utilisateur doit être au moins niveau 10 pour rejoindre une union.";
            } else {
                $existing_member = db_get_one(
                    "SELECT id FROM union_members WHERE union_id = ? AND user_id = ?",
                    [$user_union['id'], $target_user['id']]
                );
                
                if ($existing_member) {
                    $error = "Cet utilisateur est déjà membre de l'union.";
                } else {
                    $existing_invite = db_get_one(
                        "SELECT id FROM union_invitations WHERE union_id = ? AND user_id = ? AND status = 'pending'",
                        [$user_union['id'], $target_user['id']]
                    );
                    
                    if ($existing_invite) {
                        $error = "Cet utilisateur a déjà une invitation en attente.";
                    } else {
                        $invitation_data = [
                            'union_id' => $user_union['id'],
                            'user_id' => $target_user['id'],
                            'invited_by' => $_SESSION['user_id'],
                            'status' => 'pending',
                            'created_at' => date('Y-m-d H:i:s')
                        ];
                        
                        db_insert('union_invitations', $invitation_data);
                        $success = "Invitation envoyée à " . $username . ".";
                    }
                }
            }
        }
    }
}

// Récupérer les membres de l'union
$union_members = [];
if ($user_union) {
    $union_members = db_get_all(
        "SELECT um.*, u.username, u.level, u.last_active
         FROM union_members um
         JOIN users u ON um.user_id = u.id
         WHERE um.union_id = ?
         ORDER BY 
            CASE um.role 
                WHEN 'leader' THEN 1 
                WHEN 'officer' THEN 2 
                ELSE 3 
            END,
            u.username",
        [$user_union['id']]
    );
}

// Récupérer les unions disponibles
$available_unions = [];
if (!$user_union) {
    $available_unions = db_get_all(
        "SELECT u.*, c.name as country_name, COUNT(um.id) as member_count
         FROM unions u
         JOIN countries c ON u.country_id = c.id
         LEFT JOIN union_members um ON u.id = um.union_id
         WHERE u.country_id = ?
         GROUP BY u.id
         ORDER BY u.level DESC, member_count DESC",
        [$user['country_id']]
    );
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Unions | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <li><a href="unions.php" class="hover:text-blue-300 font-semibold">Unions</a></li>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Unions</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($user_union): ?>
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div class="lg:col-span-2">
                    <div class="card p-6 mb-6">
                        <div class="flex justify-between items-start">
                            <div>
                                <h3 class="text-xl font-semibold"><?php echo htmlspecialchars($user_union['name']); ?></h3>
                                <p class="text-sm text-gray-500">
                                    Niveau <?php echo $user_union['level']; ?> • 
                                    <?php echo htmlspecialchars($user_union['country_name']); ?> •
                                    Créée le <?php echo format_datetime($user_union['created_at']); ?>
                                </p>
                            </div>
                            <div>
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                    <?php echo ucfirst($user_union['role']); ?>
                                </span>
                            </div>
                        </div>
                        
                        <?php if ($user_union['description']): ?>
                            <div class="mt-4 p-4 bg-gray-50 rounded-lg">
                                <p><?php echo nl2br(htmlspecialchars($user_union['description'])); ?></p>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!$is_leader): ?>
                            <div class="mt-4">
                                <form method="POST" action="unions.php" onsubmit="return confirm('Es-tu sûr de vouloir quitter cette union?')">
                                    <input type="hidden" name="action" value="leave_union">
                                    <button type="submit" class="text-red-600 hover:text-red-800 font-medium">
                                        Quitter l'union
                                    </button>
                                </form>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="card p-6">
                        <h3 class="text-lg font-semibold mb-4">Membres (<?php echo count($union_members); ?>)</h3>
                        
                        <div class="space-y-4">
                            <?php foreach ($union_members as $member): ?>
                                <div class="border rounded-lg p-4 flex justify-between items-center">
                                    <div>
                                        <h4 class="font-medium"><?php echo htmlspecialchars($member['username']); ?></h4>
                                        <div class="text-sm text-gray-500">
                                            Niveau <?php echo $member['level']; ?> • 
                                            <?php 
                                                $last_active = new DateTime($member['last_active']);
                                                $now = new DateTime();
                                                $diff = $now->diff($last_active);
                                                
                                                if ($diff->days > 0) {
                                                    echo "Actif il y a " . $diff->days . " jour" . ($diff->days > 1 ? 's' : '');
                                                } elseif ($diff->h > 0) {
                                                    echo "Actif il y a " . $diff->h . " heure" . ($diff->h > 1 ? 's' : '');
                                                } else {
                                                    echo "Actif récemment";
                                                }
                                            ?>
                                        </div>
                                    </div>
                                    <div>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                            <?php 
                                                echo $member['role'] === 'leader' ? 'bg-purple-100 text-purple-800' : 
                                                    ($member['role'] === 'officer' ? 'bg-yellow-100 text-yellow-800' : 
                                                    'bg-gray-100 text-gray-800');
                                            ?>">
                                            <?php echo ucfirst($member['role']); ?>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                
                <div>
                    <?php if ($is_leader || $user_union['role'] === 'officer'): ?>
                        <div class="card p-6 mb-6">
                            <h3 class="text-lg font-semibold mb-4">Inviter un membre</h3>
                            
                            <form method="POST" action="unions.php" class="space-y-4">
                                <input type="hidden" name="action" value="invite_member">
                                
                                <div>
                                    <label for="username" class="block text-sm font-medium mb-1">Nom d'utilisateur</label>
                                    <input type="text" id="username" name="username" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                </div>
                                
                                <div>
                                    <button type="submit" class="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                        Envoyer l'invitation
                                    </button>
                                </div>
                            </form>
                        </div>
                    <?php endif; ?>
                    
                    <div class="card p-6">
                        <h3 class="text-lg font-semibold mb-4">Avantages de l'union</h3>
                        
                        <div class="space-y-3">
                            <div>
                                <p class="font-medium">Communications sécurisées</p>
                                <p class="text-sm text-gray-600">Échange d'informations cryptées entre les membres</p>
                            </div>
                            <div>
                                <p class="font-medium">Missions conjointes</p>
                                <p class="text-sm text-gray-600">Accès à des missions spéciales réservées aux unions</p>
                            </div>
                            <div>
                                <p class="font-medium">Bonus de défense</p>
                                <p class="text-sm text-gray-600">Protection accrue du système national</p>
                            </div>
                            <div>
                                <p class="font-medium">Base d'opérations</p>
                                <p class="text-sm text-gray-600">Lieu sécurisé pour planifier des stratégies</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div class="lg:col-span-2">
                    <div class="card p-6 mb-6">
                        <h3 class="text-lg font-semibold mb-4">Créer une nouvelle union</h3>
                        
                        <form method="POST" action="unions.php" class="space-y-4">
                            <input type="hidden" name="action" value="create_union">
                            
                            <div>
                                <label for="name" class="block text-sm font-medium mb-1">Nom de l'union</label>
                                <input type="text" id="name" name="name" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                            </div>
                            
                            <div>
                                <label for="description" class="block text-sm font-medium mb-1">Description</label>
                                <textarea id="description" name="description" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
                            </div>
                            
                            <div>
                                <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                    Créer l'union
                                </button>
                            </div>
                        </form>
                    </div>
                    
                    <div class="card p-6">
                        <h3 class="text-lg font-semibold mb-4">Unions disponibles</h3>
                        
                        <?php if (empty($available_unions)): ?>
                            <p class="text-gray-500 italic">Aucune union n'est disponible dans ton pays. Crée la première!</p>
                        <?php else: ?>
                            <div class="space-y-4">
                                <?php foreach ($available_unions as $union): ?>
                                    <div class="border rounded-lg p-4">
                                        <div class="flex justify-between items-start">
                                            <div>
                                                <h4 class="font-medium"><?php echo htmlspecialchars($union['name']); ?></h4>
                                                <p class="text-sm text-gray-500">
                                                    Niveau <?php echo $union['level']; ?> • 
                                                    <?php echo $union['member_count']; ?> membre<?php echo $union['member_count'] > 1 ? 's' : ''; ?> •
                                                    Créée le <?php echo format_datetime($union['created_at']); ?>
                                                </p>
                                            </div>
                                            <div>
                                                <a href="join_union.php?id=<?php echo $union['id']; ?>" class="inline-block bg-blue-600 hover:bg-blue-700 text-white text-sm py-1 px-3 rounded transition-colors">
                                                    Rejoindre
                                                </a>
                                            </div>
                                        </div>
                                        
                                        <?php if ($union['description']): ?>
                                            <div class="mt-3 text-sm">
                                                <p><?php echo nl2br(htmlspecialchars($union['description'])); ?></p>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div>
                    <div class="card p-6">
                        <h3 class="text-lg font-semibold mb-4">Avantages des unions</h3>
                        
                        <div class="space-y-3">
                            <div>
                            <p class="font-medium">Communications sécurisées</p>
                                <p class="text-sm text-gray-600">Échange d'informations cryptées entre les membres</p>
                            </div>
                            <div>
                                <p class="font-medium">Missions conjointes</p>
                                <p class="text-sm text-gray-600">Accès à des missions spéciales réservées aux unions</p>
                            </div>
                            <div>
                                <p class="font-medium">Bonus de défense</p>
                                <p class="text-sm text-gray-600">Protection accrue du système national</p>
                            </div>
                            <div>
                                <p class="font-medium">Base d'opérations</p>
                                <p class="text-sm text-gray-600">Lieu sécurisé pour planifier des stratégies</p>
                            </div>
                            <div>
                                <p class="font-medium">Partage de ressources</p>
                                <p class="text-sm text-gray-600">Possibilité de partager des objets et des devises</p>
                            </div>
                        </div>
                        
                        <div class="mt-6 p-4 bg-blue-50 rounded-lg">
                            <p class="text-blue-800 font-medium">Niveau requis: 10</p>
                            <p class="text-sm text-gray-600 mt-1">Pour préserver la qualité des unions, seuls les agents expérimentés peuvent y participer</p>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </main>
</body>
</html>