<?php
session_start();

function is_logged_in() {
    return isset($_SESSION['user_id']);
}

function require_login() {
    if (!is_logged_in()) {
        header('Location: login.php');
        exit;
    }
}

function get_user_info($user_id) {
    return db_get_one("SELECT * FROM users WHERE id = ?", [$user_id]);
}

function get_user_country($user_id) {
    return db_get_one("
        SELECT c.* 
        FROM countries c
        JOIN users u ON c.id = u.country_id
        WHERE u.id = ?
    ", [$user_id]);
}

function get_nearby_users($lat, $lng, $radius_km = 10, $exclude_user_id = null) {
    $earth_radius = 6371;
    
    $sql = "
        SELECT 
            u.*,
            ($earth_radius * acos(
                cos(radians(?)) * cos(radians(last_position_lat)) * 
                cos(radians(last_position_lng) - radians(?)) + 
                sin(radians(?)) * sin(radians(last_position_lat))
            )) AS distance
        FROM users u
        WHERE 
            last_position_lat IS NOT NULL AND 
            last_position_lng IS NOT NULL
    ";
    
    $params = [$lat, $lng, $lat];
    
    if ($exclude_user_id) {
        $sql .= " AND u.id != ?";
        $params[] = $exclude_user_id;
    }
    
    $sql .= " HAVING distance < ? ORDER BY distance";
    $params[] = $radius_km;
    
    return db_get_all($sql, $params);
}

function format_datetime($datetime) {
    $date = new DateTime($datetime);
    return $date->format('d/m/Y H:i');
}

function get_user_level_info($level) {
    $levels = [
        1 => ['min_exp' => 0, 'max_exp' => 100, 'title' => 'Recrue'],
        2 => ['min_exp' => 100, 'max_exp' => 300, 'title' => 'Éclaireur'],
        3 => ['min_exp' => 300, 'max_exp' => 600, 'title' => 'Agent'],
        4 => ['min_exp' => 600, 'max_exp' => 1000, 'title' => 'Agent confirmé'],
        5 => ['min_exp' => 1000, 'max_exp' => 1500, 'title' => 'Vétéran'],
        6 => ['min_exp' => 1500, 'max_exp' => 2100, 'title' => 'Expert'],
        7 => ['min_exp' => 2100, 'max_exp' => 2800, 'title' => 'Élite'],
        8 => ['min_exp' => 2800, 'max_exp' => 3600, 'title' => 'Commandant'],
        9 => ['min_exp' => 3600, 'max_exp' => 4500, 'title' => 'Stratège'],
        10 => ['min_exp' => 4500, 'max_exp' => 5500, 'title' => 'Maître'],
        11 => ['min_exp' => 5500, 'max_exp' => 6600, 'title' => 'Grand Maître'],
        12 => ['min_exp' => 6600, 'max_exp' => 7800, 'title' => 'Légende'],
        13 => ['min_exp' => 7800, 'max_exp' => 9100, 'title' => 'Mythique'],
        14 => ['min_exp' => 9100, 'max_exp' => 10500, 'title' => 'Transcendant'],
        15 => ['min_exp' => 10500, 'max_exp' => 12000, 'title' => 'Souverain'],
    ];
    
    if (isset($levels[$level])) {
        return $levels[$level];
    } else if ($level > 15) {
        $last_level = $levels[15];
        $min_exp = $last_level['max_exp'];
        $max_exp = $min_exp + 1500 * ($level - 15);
        return [
            'min_exp' => $min_exp,
            'max_exp' => $max_exp,
            'title' => 'Souverain niveau ' . ($level - 14)
        ];
    }
    
    return $levels[1];
}

function calculate_travel_time($start_lat, $start_lng, $end_lat, $end_lng, $transport_type) {
    $earth_radius = 6371;
    
    $lat1 = deg2rad($start_lat);
    $lon1 = deg2rad($start_lng);
    $lat2 = deg2rad($end_lat);
    $lon2 = deg2rad($end_lng);
    
    $dlon = $lon2 - $lon1;
    $dlat = $lat2 - $lat1;
    
    $a = sin($dlat/2) * sin($dlat/2) + cos($lat1) * cos($lat2) * sin($dlon/2) * sin($dlon/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    $distance = $earth_radius * $c;
    
    $speeds = [
        'car' => 90,    
        'train' => 180,  
        'plane' => 800,  
        'boat' => 35     
    ];
    
    $speed = $speeds[$transport_type] ?? $speeds['car'];
    
    $hours = $distance / $speed;
    
    return ceil($hours * 60);
}

function add_experience($user_id, $amount) {
    $user = get_user_info($user_id);
    $new_exp = $user['experience'] + $amount;
    
    $current_level_info = get_user_level_info($user['level']);
    $max_exp = $current_level_info['max_exp'];
    $new_level = $user['level'];
    
    while ($new_exp >= $max_exp) {
        $new_level++;
        $new_level_info = get_user_level_info($new_level);
        $max_exp = $new_level_info['max_exp'];
    }
    
    $data = [
        'experience' => $new_exp,
        'level' => $new_level
    ];
    
    db_update('users', $data, 'id = ?', [$user_id]);
    
    return $new_level > $user['level'];
}

function generate_random_combination() {
    return str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
}

function check_system_vulnerability($country_id, $combination) {
    $exists = db_get_value(
        "SELECT COUNT(*) FROM system_combinations WHERE country_id = ? AND combination = ? AND active = 1",
        [$country_id, $combination]
    );
    
    return $exists == 0;
}

function add_system_combination($country_id, $combination, $user_id) {
    $data = [
        'country_id' => $country_id,
        'combination' => $combination,
        'added_by' => $user_id,
        'active' => 1,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    return db_insert('system_combinations', $data);
}

function attack_country_system($attacker_id, $country_id, $combination) {
    $is_vulnerable = check_system_vulnerability($country_id, $combination);
    
    if ($is_vulnerable) {
        $attack_strength = mt_rand(10, 30);
        
        $country = db_get_one("SELECT system_security_level FROM countries WHERE id = ?", [$country_id]);
        $new_security_level = max(1, $country['system_security_level'] - 1);
        
        db_update('countries', ['system_security_level' => $new_security_level], 'id = ?', [$country_id]);
        
        add_experience($attacker_id, 500);
        
        return [
            'success' => true,
            'message' => "Attaque réussie! Sécurité du système réduite au niveau $new_security_level.",
            'exp_gained' => 500
        ];
    }
    
    return [
        'success' => false,
        'message' => "Attaque échouée. La combinaison est déjà connue par le système de défense."
    ];
}

function get_user_inventory($user_id) {
    return db_get_all("
        SELECT i.*, iv.quantity, iv.acquired_at
        FROM inventories iv
        JOIN items i ON iv.item_id = i.id
        WHERE iv.user_id = ?
        ORDER BY i.rarity, i.name
    ", [$user_id]);
}

function get_user_currency_balance($user_id, $currency_id) {
    $received = db_get_value("
        SELECT COALESCE(SUM(amount), 0) 
        FROM economy_transactions 
        WHERE receiver_id = ? AND currency_id = ?
    ", [$user_id, $currency_id]);
    
    $sent = db_get_value("
        SELECT COALESCE(SUM(amount), 0) 
        FROM economy_transactions 
        WHERE sender_id = ? AND currency_id = ?
    ", [$user_id, $currency_id]);
    
    return $received - $sent;
}

function get_active_world_events() {
    $now = date('Y-m-d H:i:s');
    return db_get_all("
        SELECT * FROM world_events
        WHERE start_time <= ? AND (end_time IS NULL OR end_time >= ?)
        ORDER BY start_time DESC
    ", [$now, $now]);
}

function get_available_missions($user_id) {
    $user = get_user_info($user_id);
    
    return db_get_all("
        SELECT m.*
        FROM missions m
        LEFT JOIN user_missions um ON m.id = um.mission_id AND um.user_id = ?
        WHERE 
            m.required_level <= ? AND
            (um.id IS NULL OR um.status = 'failed') AND
            (m.expiry_date IS NULL OR m.expiry_date > NOW())
        ORDER BY m.difficulty, m.required_level
    ", [$user_id, $user['level']]);
}

function get_nearby_unions($country_id) {
    return db_get_all("
        SELECT u.*, COUNT(um.id) as member_count
        FROM unions u
        LEFT JOIN union_members um ON u.id = um.union_id
        WHERE u.country_id = ?
        GROUP BY u.id
        ORDER BY u.level DESC, member_count DESC
    ", [$country_id]);
}

function get_user_alliances($user_id) {
    return db_get_all("
        SELECT a.*, 
            u.username as ally_username,
            u.level as ally_level
        FROM alliances a
        JOIN users u ON (a.user1_id = ? AND a.user2_id = u.id) OR (a.user2_id = ? AND a.user1_id = u.id)
        WHERE a.status = 'active'
        ORDER BY u.username
    ", [$user_id, $user_id]);
}

function send_message($sender_id, $receiver_id, $content, $type = 'text') {
    $data = [
        'sender_id' => $sender_id,
        'receiver_id' => $receiver_id,
        'content' => $content,
        'type' => $type,
        'read' => 0,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    return db_insert('communications', $data);
}

function update_user_position($user_id, $lat, $lng) {
    return db_update('users', 
        ['last_position_lat' => $lat, 'last_position_lng' => $lng, 'last_active' => date('Y-m-d H:i:s')],
        'id = ?',
        [$user_id]
    );
}

function start_travel($user_id, $start_country, $end_country, $start_coords, $end_coords, $transport_type) {
    list($start_lat, $start_lng) = explode(',', $start_coords);
    list($end_lat, $end_lng) = explode(',', $end_coords);
    
    $travel_minutes = calculate_travel_time($start_lat, $start_lng, $end_lat, $end_lng, $transport_type);
    $arrival_time = date('Y-m-d H:i:s', strtotime("+$travel_minutes minutes"));
    
    $data = [
        'user_id' => $user_id,
        'start_country' => $start_country,
        'end_country' => $end_country,
        'start_coordinates' => $start_coords,
        'end_coordinates' => $end_coords,
        'start_time' => date('Y-m-d H:i:s'),
        'arrival_time' => $arrival_time,
        'status' => 'in_progress',
        'transport_type' => $transport_type
    ];
    
    return [
        'travel_id' => db_insert('travels', $data),
        'arrival_time' => $arrival_time,
        'travel_minutes' => $travel_minutes
    ];
}

function is_traveling($user_id) {
    return db_get_one("
        SELECT * FROM travels 
        WHERE user_id = ? AND status = 'in_progress' 
        ORDER BY start_time DESC 
        LIMIT 1
    ", [$user_id]);
}

function get_random_mission($user_level, $country_id = null) {
    $difficulty_tiers = ['easy', 'medium', 'hard', 'expert'];
    $difficulties = [];
    
    if ($user_level < 5) {
        $difficulties[] = 'easy';
    } elseif ($user_level < 10) {
        $difficulties[] = 'easy';
        $difficulties[] = 'medium';
    } elseif ($user_level < 15) {
        $difficulties[] = 'medium';
        $difficulties[] = 'hard';
    } else {
        $difficulties[] = 'hard';
        $difficulties[] = 'expert';
    }
    
    $difficulty_in = implode("','", $difficulties);
    
    $sql = "
        SELECT * FROM missions 
        WHERE difficulty IN ('$difficulty_in')
        AND required_level <= ?
    ";
    
    $params = [$user_level];
    
    if ($country_id) {
        $sql .= " AND (country_id = ? OR country_id IS NULL)";
        $params[] = $country_id;
    }
    
    $sql .= " ORDER BY RAND() LIMIT 1";
    
    return db_get_one($sql, $params);
}

function use_item($user_id, $item_id) {
    $item = db_get_one("SELECT * FROM items WHERE id = ?", [$item_id]);
    
    if (!$item) {
        return [
            'success' => false,
            'message' => "Objet non trouvé."
        ];
    }
    
    $inventory_item = db_get_one(
        "SELECT * FROM inventories WHERE user_id = ? AND item_id = ?",
        [$user_id, $item_id]
    );
    
    if (!$inventory_item) {
        return [
            'success' => false,
            'message' => "Tu ne possèdes pas cet objet."
        ];
    }
    
    if ($inventory_item['quantity'] <= 0) {
        return [
            'success' => false,
            'message' => "Quantité insuffisante."
        ];
    }
    
    // Logique d'utilisation des objets selon leur type
    switch ($item['type']) {
        case 'équipement':
            // Les équipements ne se consomment pas
            return [
                'success' => true,
                'message' => "Tu as équipé: " . $item['name'] . "."
            ];
            break;
            
        case 'consommable':
            // Réduire la quantité de 1
            if ($inventory_item['quantity'] == 1) {
                db_delete('inventories', 'id = ?', [$inventory_item['id']]);
            } else {
                db_update(
                    'inventories',
                    ['quantity' => $inventory_item['quantity'] - 1],
                    'id = ?',
                    [$inventory_item['id']]
                );
            }
            
            // Effet spécial selon l'objet
            if ($item['id'] == 4) { // Passe-partout numérique
                return [
                    'success' => true,
                    'message' => "Tu as utilisé: " . $item['name'] . ". Tu peux maintenant accéder à un système de niveau 3 ou inférieur.",
                    'redirect' => 'system.php'
                ];
            }
            
            return [
                'success' => true,
                'message' => "Tu as utilisé: " . $item['name'] . "."
            ];
            break;
            
        default:
            return [
                'success' => false,
                'message' => "Cet objet ne peut pas être utilisé directement."
            ];
    }
}