<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$current_tab = isset($_GET['tab']) ? $_GET['tab'] : 'messages';

$error = null;
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'send_message') {
        $receiver_id = $_POST['receiver_id'] ?? '';
        $content = trim($_POST['content'] ?? '');
        
        if (empty($receiver_id)) {
            $error = "Le destinataire est requis.";
        } elseif (empty($content)) {
            $error = "Le message ne peut pas être vide.";
        } elseif ($receiver_id == $_SESSION['user_id']) {
            $error = "Tu ne peux pas t'envoyer un message à toi-même.";
        } else {
            $message_id = send_message($_SESSION['user_id'], $receiver_id, $content);
            
            if ($message_id) {
                $success = "Message envoyé avec succès.";
            } else {
                $error = "Erreur lors de l'envoi du message.";
            }
        }
    } elseif (isset($_POST['action']) && $_POST['action'] === 'alliance_response') {
        $alliance_id = $_POST['alliance_id'] ?? '';
        $response = $_POST['response'] ?? '';
        
        if (empty($alliance_id) || empty($response)) {
            $error = "Informations manquantes.";
        } else {
            $status = ($response === 'accept') ? 'active' : 'rejected';
            
            $updated = db_update(
                'alliances',
                ['status' => $status],
                'id = ? AND user2_id = ?',
                [$alliance_id, $_SESSION['user_id']]
            );
            
            if ($updated) {
                $success = ($response === 'accept') ? "Alliance acceptée." : "Alliance rejetée.";
            } else {
                $error = "Erreur lors du traitement de la demande d'alliance.";
            }
        }
    }
}

$received_messages = db_get_all(
    "SELECT c.*, u.username as sender_username
     FROM communications c
     JOIN users u ON c.sender_id = u.id
     WHERE c.receiver_id = ?
     ORDER BY c.created_at DESC
     LIMIT 50",
    [$_SESSION['user_id']]
);

$sent_messages = db_get_all(
    "SELECT c.*, u.username as receiver_username
     FROM communications c
     JOIN users u ON c.receiver_id = u.id
     WHERE c.sender_id = ?
     ORDER BY c.created_at DESC
     LIMIT 50",
    [$_SESSION['user_id']]
);

$alliances = db_get_all(
    "SELECT a.*, 
        CASE
            WHEN a.user1_id = ? THEN u2.username
            ELSE u1.username
        END as ally_username,
        CASE
            WHEN a.user1_id = ? THEN u2.id
            ELSE u1.id
        END as ally_id
     FROM alliances a
     JOIN users u1 ON a.user1_id = u1.id
     JOIN users u2 ON a.user2_id = u2.id
     WHERE (a.user1_id = ? OR a.user2_id = ?)
     ORDER BY a.created_at DESC",
    [$_SESSION['user_id'], $_SESSION['user_id'], $_SESSION['user_id'], $_SESSION['user_id']]
);

$pending_alliances = array_filter($alliances, function($a) {
    return $a['status'] === 'pending' && $a['user2_id'] === $_SESSION['user_id'];
});

$active_alliances = array_filter($alliances, function($a) {
    return $a['status'] === 'active';
});

$allies = db_get_all(
    "SELECT u.id, u.username, u.level, u.last_active
     FROM users u
     JOIN alliances a ON (a.user1_id = u.id OR a.user2_id = u.id)
     WHERE a.status = 'active'
     AND (
         (a.user1_id = ? AND a.user2_id = u.id)
         OR
         (a.user2_id = ? AND a.user1_id = u.id)
     )
     ORDER BY u.username",
    [$_SESSION['user_id'], $_SESSION['user_id']]
);

$nearby_users = [];
if ($user['last_position_lat'] && $user['last_position_lng']) {
    $nearby_users = get_nearby_users(
        $user['last_position_lat'],
        $user['last_position_lng'],
        10,
        $_SESSION['user_id']
    );
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Communications | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
        .tab-active {
            border-bottom: 2px solid #3B82F6;
            color: #3B82F6;
        }
        .message-bubble-received {
            background-color: #F3F4F6;
            border-radius: 18px 18px 18px 4px;
        }
        .message-bubble-sent {
            background-color: #DBEAFE;
            border-radius: 18px 18px 4px 18px;
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300 font-semibold">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Communications</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($pending_alliances)): ?>
            <div class="card p-4 mb-6">
                <h3 class="text-lg font-semibold mb-2">Demandes d'alliance en attente</h3>
                <div class="space-y-3">
                    <?php foreach($pending_alliances as $alliance): ?>
                        <div class="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                            <div>
                                <p class="font-medium"><?php echo htmlspecialchars($alliance['ally_username']); ?> souhaite former une alliance avec toi</p>
                                <p class="text-sm text-gray-500">Demande envoyée le <?php echo format_datetime($alliance['created_at']); ?></p>
                            </div>
                            <div class="flex space-x-2">
                                <form method="POST" action="communication.php?tab=<?php echo $current_tab; ?>" class="inline-block">
                                    <input type="hidden" name="action" value="alliance_response">
                                    <input type="hidden" name="alliance_id" value="<?php echo $alliance['id']; ?>">
                                    <input type="hidden" name="response" value="accept">
                                    <button type="submit" class="bg-green-600 hover:bg-green-700 text-white py-1 px-3 rounded">Accepter</button>
                                </form>
                                <form method="POST" action="communication.php?tab=<?php echo $current_tab; ?>" class="inline-block">
                                    <input type="hidden" name="action" value="alliance_response">
                                    <input type="hidden" name="alliance_id" value="<?php echo $alliance['id']; ?>">
                                    <input type="hidden" name="response" value="reject">
                                    <button type="submit" class="bg-red-600 hover:bg-red-700 text-white py-1 px-3 rounded">Rejeter</button>
                                </form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <div class="card mb-6">
            <div class="border-b">
                <div class="flex">
                    <a href="communication.php?tab=messages" class="px-4 py-3 text-center flex-1 <?php echo $current_tab === 'messages' ? 'tab-active font-medium' : 'text-gray-600 hover:bg-gray-50'; ?>">
                        Messages
                    </a>
                    <a href="communication.php?tab=allies" class="px-4 py-3 text-center flex-1 <?php echo $current_tab === 'allies' ? 'tab-active font-medium' : 'text-gray-600 hover:bg-gray-50'; ?>">
                        Alliés (<?php echo count($active_alliances); ?>)
                    </a>
                    <a href="communication.php?tab=nearby" class="px-4 py-3 text-center flex-1 <?php echo $current_tab === 'nearby' ? 'tab-active font-medium' : 'text-gray-600 hover:bg-gray-50'; ?>">
                        À proximité (<?php echo count($nearby_users); ?>)
                    </a>
                </div>
            </div>
            
            <div class="p-6">
                <?php if ($current_tab === 'messages'): ?>
                    <div class="mb-6">
                        <h3 class="text-lg font-semibold mb-4">Nouveau message</h3>
                        
                        <form method="POST" action="communication.php" class="space-y-4">
                            <input type="hidden" name="action" value="send_message">
                            
                            <div>
                                <label for="receiver_id" class="block text-sm font-medium mb-1">Destinataire</label>
                                <select id="receiver_id" name="receiver_id" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Sélectionne un destinataire</option>
                                    <optgroup label="Alliés">
                                        <?php foreach ($allies as $ally): ?>
                                            <option value="<?php echo $ally['id']; ?>">
                                                <?php echo htmlspecialchars($ally['username']); ?> (Niveau <?php echo $ally['level']; ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                    <?php if (!empty($nearby_users)): ?>
                                        <optgroup label="Agents à proximité">
                                            <?php foreach ($nearby_users as $nearby): ?>
                                                <option value="<?php echo $nearby['id']; ?>">
                                                    <?php echo htmlspecialchars($nearby['username']); ?> (Niveau <?php echo $nearby['level']; ?>)
                                                </option>
                                            <?php endforeach; ?>
                                        </optgroup>
                                    <?php endif; ?>
                                </select>
                            </div>
                            
                            <div>
                                <label for="content" class="block text-sm font-medium mb-1">Message</label>
                                <textarea id="content" name="content" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required></textarea>
                            </div>
                            
                            <div>
                                <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                    Envoyer le message
                                </button>
                            </div>
                        </form>
                    </div>
                    
                    <div class="border-t pt-6">
                        <h3 class="text-lg font-semibold mb-4">Messages reçus</h3>
                        
                        <?php if (empty($received_messages)): ?>
                            <p class="text-gray-500 italic">Tu n'as reçu aucun message.</p>
                        <?php else: ?>
                            <div class="space-y-4">
                                <?php foreach ($received_messages as $message): ?>
                                    <div class="p-4 message-bubble-received">
                                        <div class="flex justify-between mb-1">
                                            <span class="font-medium"><?php echo htmlspecialchars($message['sender_username']); ?></span>
                                            <span class="text-sm text-gray-500"><?php echo format_datetime($message['created_at']); ?></span>
                                        </div>
                                        <div class="text-gray-800"><?php echo nl2br(htmlspecialchars($message['content'])); ?></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="border-t pt-6 mt-6">
                        <h3 class="text-lg font-semibold mb-4">Messages envoyés</h3>
                        
                        <?php if (empty($sent_messages)): ?>
                            <p class="text-gray-500 italic">Tu n'as envoyé aucun message.</p>
                        <?php else: ?>
                            <div class="space-y-4">
                                <?php foreach ($sent_messages as $message): ?>
                                    <div class="p-4 message-bubble-sent">
                                        <div class="flex justify-between mb-1">
                                            <span class="font-medium">À: <?php echo htmlspecialchars($message['receiver_username']); ?></span>
                                            <span class="text-sm text-gray-500"><?php echo format_datetime($message['created_at']); ?></span>
                                        </div>
                                        <div class="text-gray-800"><?php echo nl2br(htmlspecialchars($message['content'])); ?></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                <?php elseif ($current_tab === 'allies'): ?>
                    <h3 class="text-lg font-semibold mb-4">Tes alliés</h3>
                    
                    <?php if (empty($active_alliances)): ?>
                        <p class="text-gray-500 italic">Tu n'as pas encore d'alliés. Explore la carte pour rencontrer d'autres agents.</p>
                    <?php else: ?>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <?php foreach ($active_alliances as $alliance): ?>
                                <div class="border rounded-lg p-4 hover:bg-gray-50">
                                    <div class="flex justify-between items-start">
                                        <div>
                                            <h4 class="font-medium"><?php echo htmlspecialchars($alliance['ally_username']); ?></h4>
                                            <p class="text-sm text-gray-500">Alliance formée le <?php echo format_datetime($alliance['created_at']); ?></p>
                                        </div>
                                        <div class="flex space-x-2">
                                            <a href="communication.php?tab=messages" class="text-blue-600 hover:underline text-sm">
                                                Envoyer un message
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                    
                <?php elseif ($current_tab === 'nearby'): ?>
                    <h3 class="text-lg font-semibold mb-4">Agents à proximité</h3>
                    
                    <?php if (empty($nearby_users)): ?>
                        <p class="text-gray-500 italic">Aucun agent détecté à proximité. Explore la carte pour rencontrer d'autres agents.</p>
                    <?php else: ?>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <?php foreach ($nearby_users as $nearby): ?>
                                <div class="border rounded-lg p-4 hover:bg-gray-50">
                                    <div class="flex justify-between items-start">
                                        <div>
                                            <h4 class="font-medium"><?php echo htmlspecialchars($nearby['username']); ?></h4>
                                            <p class="text-sm">Niveau <?php echo $nearby['level']; ?></p>
                                            <p class="text-sm text-gray-500">Distance: <?php echo round($nearby['distance'], 1); ?> km</p>
                                        </div>
                                        <div class="flex flex-col space-y-2">
                                            <a href="communication.php?tab=messages" class="text-blue-600 hover:underline text-sm">
                                                Envoyer un message
                                            </a>
                                            <?php
                                                $is_allied = false;
                                                foreach ($alliances as $alliance) {
                                                    if (($alliance['user1_id'] == $_SESSION['user_id'] && $alliance['user2_id'] == $nearby['id']) ||
                                                        ($alliance['user2_id'] == $_SESSION['user_id'] && $alliance['user1_id'] == $nearby['id'])) {
                                                        $is_allied = true;
                                                        break;
                                                    }
                                                }
                                                
                                                if (!$is_allied):
                                            ?>
                                                <form method="POST" action="request_alliance.php">
                                                    <input type="hidden" name="target_user_id" value="<?php echo $nearby['id']; ?>">
                                                    <button type="submit" class="text-green-600 hover:underline text-sm">
                                                        Demander une alliance
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                            
                                            <button type="button" class="text-red-600 hover:underline text-sm" onclick="startVoiceCall(<?php echo $nearby['id']; ?>, '<?php echo htmlspecialchars($nearby['username']); ?>')">
                                                Appel vocal
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <div id="voice-call-modal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center hidden">
        <div class="bg-white rounded-lg p-6 max-w-md w-full">
            <h3 class="text-xl font-semibold mb-4" id="call-title">Appel en cours...</h3>
            <div id="call-status" class="mb-4 text-center"></div>
            <div class="flex justify-center space-x-4">
                <button type="button" id="end-call-btn" class="bg-red-600 hover:bg-red-700 text-white py-2 px-4 rounded transition-colors">
                    Terminer l'appel
                </button>
            </div>
        </div>
    </div>

    <script>
        let websocket = null;
        let peerConnection = null;
        let localStream = null;
        let currentCallUserId = null;
        const voiceCallModal = document.getElementById('voice-call-modal');
        const callTitle = document.getElementById('call-title');
        const callStatus = document.getElementById('call-status');
        const endCallBtn = document.getElementById('end-call-btn');
        
        function connectWebSocket() {
            const wsProtocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
            const userId = <?php echo $_SESSION['user_id']; ?>;
            const token = Math.random().toString(36).substring(2, 15);
            
            websocket = new WebSocket(`${wsProtocol}//${window.location.hostname}:3000?user_id=${userId}&token=${token}`);
            
            websocket.onopen = function() {
                console.log('WebSocket connecté');
            };
            
            websocket.onmessage = function(event) {
                const message = JSON.parse(event.data);
                
                switch (message.type) {
                    case 'voice_call':
                        handleIncomingCall(message.data);
                        break;
                    case 'voice_answer':
                        handleCallAnswer(message.data);
                        break;
                }
            };
            
            websocket.onerror = function(error) {
                console.error('Erreur WebSocket:', error);
            };
            
            websocket.onclose = function() {
                console.log('WebSocket déconnecté');
                setTimeout(connectWebSocket, 5000);
            };
        }
        
        function startVoiceCall(userId, username) {
            if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
                alert('Ton navigateur ne supporte pas les appels audio.');
                return;
            }
            
            currentCallUserId = userId;
            callTitle.textContent = `Appel vers ${username}...`;
            callStatus.textContent = 'Connexion en cours...';
            voiceCallModal.classList.remove('hidden');
            
            navigator.mediaDevices.getUserMedia({ audio: true, video: false })
                .then(stream => {
                    localStream = stream;
                    
                    const configuration = { 
                        iceServers: [{ urls: 'stun:stun.l.google.com:19302' }] 
                    };
                    
                    peerConnection = new RTCPeerConnection(configuration);
                    
                    localStream.getTracks().forEach(track => {
                        peerConnection.addTrack(track, localStream);
                    });
                    
                    peerConnection.onicecandidate = event => {
                        if (event.candidate) {
                            sendVoiceCallSignal(userId, {
                                type: 'ice-candidate',
                                candidate: event.candidate
                            });
                        }
                    };
                    
                    peerConnection.ontrack = event => {
                        const remoteAudio = new Audio();
                        remoteAudio.srcObject = event.streams[0];
                        remoteAudio.play();
                    };
                    
                    peerConnection.createOffer()
                        .then(offer => peerConnection.setLocalDescription(offer))
                        .then(() => {
                            sendVoiceCallSignal(userId, {
                                type: 'offer',
                                sdp: peerConnection.localDescription
                            });
                        })
                        .catch(error => {
                            console.error('Erreur lors de la création de l\'offre:', error);
                            endCall();
                        });
                })
                .catch(error => {
                    console.error('Erreur d\'accès au microphone:', error);
                    alert('Impossible d\'accéder au microphone. Vérifie tes permissions.');
                    endCall();
                });
        }
        
        function handleIncomingCall(data) {
            if (confirm(`Appel entrant de ${data.caller_username}. Accepter?`)) {
                currentCallUserId = data.caller_id;
                callTitle.textContent = `Appel avec ${data.caller_username}`;
                callStatus.textContent = 'Connexion en cours...';
                voiceCallModal.classList.remove('hidden');
                
                navigator.mediaDevices.getUserMedia({ audio: true, video: false })
                    .then(stream => {
                        localStream = stream;
                        
                        const configuration = { 
                            iceServers: [{ urls: 'stun:stun.l.google.com:19302' }] 
                        };
                        
                        peerConnection = new RTCPeerConnection(configuration);
                        
                        localStream.getTracks().forEach(track => {
                            peerConnection.addTrack(track, localStream);
                        });
                        
                        peerConnection.onicecandidate = event => {
                            if (event.candidate) {
                                sendVoiceAnswerSignal(data.caller_id, {
                                    type: 'ice-candidate',
                                    candidate: event.candidate
                                }, true);
                            }
                        };
                        
                        peerConnection.ontrack = event => {
                            const remoteAudio = new Audio();
                            remoteAudio.srcObject = event.streams[0];
                            remoteAudio.play();
                        };
                        
                        const offerSdp = new RTCSessionDescription(data.signal_data);
                        peerConnection.setRemoteDescription(offerSdp)
                            .then(() => peerConnection.createAnswer())
                            .then(answer => peerConnection.setLocalDescription(answer))
                            .then(() => {
                                sendVoiceAnswerSignal(data.caller_id, {
                                    type: 'answer',
                                    sdp: peerConnection.localDescription
                                }, true);
                            })
                            .catch(error => {
                                console.error('Erreur lors de la création de la réponse:', error);
                                endCall();
                            });
                    })
                    .catch(error => {
                        console.error('Erreur d\'accès au microphone:', error);
                        alert('Impossible d\'accéder au microphone. Vérifie tes permissions.');
                        endCall();
                    });
            } else {
                sendVoiceAnswerSignal(data.caller_id, { type: 'rejected' }, false);
            }
        }
        
        function handleCallAnswer(data) {
            if (data.accept) {
                callStatus.textContent = 'Appel en cours...';
                
                if (data.signal_data.type === 'answer') {
                    const answerSdp = new RTCSessionDescription(data.signal_data.sdp);
                    peerConnection.setRemoteDescription(answerSdp)
                        .catch(error => {
                            console.error('Erreur lors de la définition de la description distante:', error);
                            endCall();
                        });
                } else if (data.signal_data.type === 'ice-candidate') {
                    peerConnection.addIceCandidate(new RTCIceCandidate(data.signal_data.candidate))
                        .catch(error => {
                            console.error('Erreur lors de l\'ajout du candidat ICE:', error);
                        });
                }
            } else {
                alert('Appel rejeté');
                endCall();
            }
        }
        
        function sendVoiceCallSignal(userId, signalData) {
            if (websocket && websocket.readyState === WebSocket.OPEN) {
                websocket.send(JSON.stringify({
                    type: 'voice_call',
                    data: {
                        target_user_id: userId,
                        signal_data: signalData
                    }
                }));
            }
        }
        
        function sendVoiceAnswerSignal(callerId, signalData, accept) {
            if (websocket && websocket.readyState === WebSocket.OPEN) {
                websocket.send(JSON.stringify({
                    type: 'voice_answer',
                    data: {
                        caller_id: callerId,
                        signal_data: signalData,
                        accept: accept
                    }
                }));
            }
        }
        
        function endCall() {
            if (peerConnection) {
                peerConnection.close();
                peerConnection = null;
            }
            
            if (localStream) {
                localStream.getTracks().forEach(track => track.stop());
                localStream = null;
            }
            
            currentCallUserId = null;
            voiceCallModal.classList.add('hidden');
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            connectWebSocket();
            
            endCallBtn.addEventListener('click', endCall);
        });
    </script>
</body>
</html>