<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Can - Playground</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @keyframes blink {
            0%, 49% { opacity: 1; }
            50%, 100% { opacity: 0; }
        }
        .cursor-blink { animation: blink 1s infinite; }
        .typing { animation: typing 0.5s steps(40, end); }
        @keyframes typing {
            from { width: 0; }
            to { width: 100%; }
        }
        .file-icon::before {
            content: '📄';
            margin-right: 8px;
        }
        .folder-icon::before {
            content: '📁';
            margin-right: 8px;
        }
        body {
            user-select: none;
            -webkit-user-select: none;
        }
        .terminal-text {
            user-select: text;
            -webkit-user-select: text;
        }
    </style>
</head>
<body class="bg-black text-white h-screen overflow-hidden">
    
    <div id="mobileBlock" class="hidden fixed inset-0 bg-black z-50 flex items-center justify-center p-8">
        <div class="text-center space-y-6">
            <h1 class="text-4xl font-thin">Can Playground</h1>
            <p class="text-gray-400">This playground requires a desktop computer to experience the full development environment simulation.</p>
            <a href="https://imators.systems/can" class="inline-block border border-white px-6 py-3 hover:bg-white hover:text-black transition">
                Return to Main Site
            </a>
        </div>
    </div>

    <div id="desktopApp" class="h-screen flex flex-col">
        
        <header class="bg-gray-950 border-b border-gray-900 px-6 py-3 flex items-center justify-between">
            <div class="flex items-center gap-4">
                <h1 class="text-xl font-thin">Can <span class="text-gray-500">Playground</span></h1>
                <span class="text-xs text-gray-600 px-2 py-1 bg-gray-900 rounded">DEMO MODE</span>
            </div>
            <button onclick="resetPlayground()" class="text-xs text-gray-500 hover:text-white transition px-3 py-1 border border-gray-800 hover:border-gray-600 rounded">
                Reset Environment
            </button>
        </header>

        <div id="welcomeScreen" class="flex-1 flex items-center justify-center p-8">
            <div class="max-w-2xl text-center space-y-6">
                <h2 class="text-3xl font-thin mb-8">Welcome to Can Playground</h2>
                <p class="text-gray-400 leading-relaxed">
                    This is a simulated environment showcasing Can's capabilities. 
                    The algorithm has been adapted for web demonstration purposes.
                    All operations are virtual and will reset on page refresh.
                </p>
                <div class="text-sm text-gray-500 space-y-2">
                    <p>• Select predefined commands from the panel</p>
                    <p>• Watch Can create projects in real-time</p>
                    <p>• Explore generated files and folders</p>
                    <p>• Everything runs in your browser memory</p>
                </div>
                <button onclick="startPlayground()" class="mt-8 px-8 py-3 bg-white text-black hover:bg-gray-200 transition">
                    Start Playground
                </button>
            </div>
        </div>

        <div id="mainInterface" class="hidden flex-1 flex">
            
            <aside class="w-64 bg-gray-950 border-r border-gray-900 flex flex-col">
                <div class="p-4 border-b border-gray-900">
                    <h3 class="text-xs text-gray-500 uppercase tracking-wider mb-1">Project</h3>
                    <div id="projectName" class="font-mono text-sm">~/playground</div>
                </div>
                <div id="fileTree" class="flex-1 overflow-y-auto p-4 text-sm font-mono">
                    <div class="text-gray-600 text-xs">No files yet</div>
                </div>
            </aside>

            <div class="flex-1 flex flex-col">
                
                <div id="fileViewer" class="flex-1 bg-gray-950 p-6 overflow-y-auto hidden">
                    <div class="max-w-4xl">
                        <div id="fileContent" class="font-mono text-sm text-gray-300 whitespace-pre-wrap"></div>
                    </div>
                </div>

                <div id="terminal" class="h-80 bg-black border-t border-gray-900 flex flex-col">
                    <div class="px-4 py-2 border-b border-gray-900 flex items-center justify-between">
                        <span class="text-xs text-gray-500">Terminal</span>
                        <button onclick="clearTerminal()" class="text-xs text-gray-600 hover:text-white transition">Clear</button>
                    </div>
                    <div id="terminalOutput" class="flex-1 overflow-y-auto p-4 font-mono text-sm terminal-text">
                        <div class="text-gray-500">Can Algorithm v0.1.0 (Playground Edition)</div>
                        <div class="text-gray-500">Ready for commands...</div>
                        <div class="mt-2">
                            <span class="text-gray-600">$</span>
                            <span id="currentCommand" class="ml-2"></span>
                            <span class="cursor-blink">_</span>
                        </div>
                    </div>
                </div>
            </div>

            <aside class="w-80 bg-gray-950 border-l border-gray-900 p-4 overflow-y-auto">
                <h3 class="text-xs text-gray-500 uppercase tracking-wider mb-4">Available Commands</h3>
                <div class="space-y-2">
                    <button onclick="executeCommand('can proj \"simple website\"')" class="command-btn w-full text-left p-3 bg-gray-900 hover:bg-gray-800 transition rounded text-sm">
                        <div class="font-mono text-xs text-blue-400 mb-1">can proj "simple website"</div>
                        <div class="text-xs text-gray-500">Creates a basic HTML/CSS site</div>
                    </button>
                    
                    <button onclick="executeCommand('can proj \"react dashboard\"')" class="command-btn w-full text-left p-3 bg-gray-900 hover:bg-gray-800 transition rounded text-sm">
                        <div class="font-mono text-xs text-green-400 mb-1">can proj "react dashboard"</div>
                        <div class="text-xs text-gray-500">React app with components</div>
                    </button>
                    
                    <button onclick="executeCommand('can proj \"e-commerce php\"')" class="command-btn w-full text-left p-3 bg-gray-900 hover:bg-gray-800 transition rounded text-sm">
                        <div class="font-mono text-xs text-purple-400 mb-1">can proj "e-commerce php"</div>
                        <div class="text-xs text-gray-500">PHP shop with cart system</div>
                    </button>
                    
                    <button onclick="executeCommand('can proj \"api rest node\"')" class="command-btn w-full text-left p-3 bg-gray-900 hover:bg-gray-800 transition rounded text-sm">
                        <div class="font-mono text-xs text-yellow-400 mb-1">can proj "api rest node"</div>
                        <div class="text-xs text-gray-500">Node.js REST API</div>
                    </button>
                    
                    <button onclick="executeCommand('can status')" class="command-btn w-full text-left p-3 bg-gray-900 hover:bg-gray-800 transition rounded text-sm">
                        <div class="font-mono text-xs text-gray-400 mb-1">can status</div>
                        <div class="text-xs text-gray-500">Check system status</div>
                    </button>
                    
                    <button onclick="executeCommand('can scan')" class="command-btn w-full text-left p-3 bg-gray-900 hover:bg-gray-800 transition rounded text-sm">
                        <div class="font-mono text-xs text-cyan-400 mb-1">can scan</div>
                        <div class="text-xs text-gray-500">Analyze current project</div>
                    </button>
                </div>
                
                <div class="mt-6 p-3 bg-gray-900 rounded text-xs text-gray-500">
                    <p class="mb-2">💡 Tip: Click any command to execute</p>
                    <p>Files created are virtual and stored in browser memory only.</p>
                </div>
            </aside>
        </div>
    </div>

    <script>
        let virtualFS = {};
        let currentPath = '~/playground';
        let isExecuting = false;

        function checkDevice() {
            if (window.innerWidth < 1024) {
                document.getElementById('mobileBlock').classList.remove('hidden');
                document.getElementById('desktopApp').classList.add('hidden');
            }
        }

        window.addEventListener('load', checkDevice);
        window.addEventListener('resize', checkDevice);

        function startPlayground() {
            document.getElementById('welcomeScreen').classList.add('hidden');
            document.getElementById('mainInterface').classList.remove('hidden');
        }

        function resetPlayground() {
            virtualFS = {};
            document.getElementById('fileTree').innerHTML = '<div class="text-gray-600 text-xs">No files yet</div>';
            document.getElementById('terminalOutput').innerHTML = `
                <div class="text-gray-500">Can Algorithm v0.1.0 (Playground Edition)</div>
                <div class="text-gray-500">Environment reset successfully.</div>
                <div class="mt-2">
                    <span class="text-gray-600">$</span>
                    <span id="currentCommand" class="ml-2"></span>
                    <span class="cursor-blink">_</span>
                </div>
            `;
            document.getElementById('fileViewer').classList.add('hidden');
        }

        function clearTerminal() {
            document.getElementById('terminalOutput').innerHTML = `
                <div class="mt-2">
                    <span class="text-gray-600">$</span>
                    <span id="currentCommand" class="ml-2"></span>
                    <span class="cursor-blink">_</span>
                </div>
            `;
        }

        async function executeCommand(command) {
            if (isExecuting) return;
            isExecuting = true;

            const buttons = document.querySelectorAll('.command-btn');
            buttons.forEach(btn => btn.disabled = true);

            const terminal = document.getElementById('terminalOutput');
            terminal.innerHTML = terminal.innerHTML.replace('<span class="cursor-blink">_</span>', '');
            
            let typedCommand = '';
            for (let char of command) {
                typedCommand += char;
                terminal.innerHTML = terminal.innerHTML.replace(/<span id="currentCommand".*<\/span>/, 
                    `<span id="currentCommand" class="ml-2">${typedCommand}</span>`);
                await sleep(30);
            }
            
            terminal.innerHTML += '\n';
            
            if (command.includes('proj') && command.includes('simple website')) {
                await createSimpleWebsite();
            } else if (command.includes('proj') && command.includes('react dashboard')) {
                await createReactDashboard();
            } else if (command.includes('proj') && command.includes('e-commerce php')) {
                await createEcommercePHP();
            } else if (command.includes('proj') && command.includes('api rest node')) {
                await createNodeAPI();
            } else if (command === 'can status') {
                await showStatus();
            } else if (command === 'can scan') {
                await scanProject();
            }

            terminal.innerHTML += `
                <div class="mt-2">
                    <span class="text-gray-600">$</span>
                    <span id="currentCommand" class="ml-2"></span>
                    <span class="cursor-blink">_</span>
                </div>
            `;
            terminal.scrollTop = terminal.scrollHeight;

            buttons.forEach(btn => btn.disabled = false);
            isExecuting = false;
        }

        async function createSimpleWebsite() {
            await addTerminalLine('🚀 Initializing simple website project...', 'text-green-400');
            await sleep(500);
            await addTerminalLine('📁 Creating project structure...');
            
            virtualFS['index.html'] = `<!DOCTYPE html>
<html>
<head>
    <title>Simple Website</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <h1>Welcome to Can Generated Site</h1>
    <p>This website was created automatically by Can Algorithm.</p>
</body>
</html>`;
            
            virtualFS['style.css'] = `body {
    font-family: system-ui;
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem;
}

h1 {
    color: #333;
}`;
            
            virtualFS['script.js'] = `console.log('Can Algorithm Generated This');`;
            
            await sleep(300);
            await addTerminalLine('✅ Project created successfully!', 'text-green-400');
            updateFileTree();
        }

        async function createReactDashboard() {
            await addTerminalLine('🚀 Creating React dashboard...', 'text-green-400');
            await sleep(500);
            
            virtualFS['package.json'] = `{
  "name": "react-dashboard",
  "version": "1.0.0",
  "dependencies": {
    "react": "^18.0.0",
    "react-dom": "^18.0.0"
  }
}`;
            
            virtualFS['App.jsx'] = `import React from 'react';

function App() {
    return (
        <div className="dashboard">
            <h1>Dashboard</h1>
            <div className="stats">
                <div>Users: 1,234</div>
                <div>Revenue: $45,678</div>
            </div>
        </div>
    );
}

export default App;`;
            
            virtualFS['index.js'] = `import React from 'react';
import ReactDOM from 'react-dom';
import App from './App';

ReactDOM.render(<App />, document.getElementById('root'));`;
            
            await sleep(300);
            await addTerminalLine('✅ React dashboard ready!', 'text-green-400');
            updateFileTree();
        }

        async function createEcommercePHP() {
            await addTerminalLine('🚀 Building e-commerce platform...', 'text-purple-400');
            await sleep(500);
            
            virtualFS['index.php'] = `<?php
session_start();
?>
<!DOCTYPE html>
<html>
<head>
    <title>E-Commerce Store</title>
</head>
<body>
    <h1>Welcome to Our Store</h1>
    <?php include 'products.php'; ?>
</body>
</html>`;
            
            virtualFS['products.php'] = `<?php
$products = [
    ['id' => 1, 'name' => 'Product 1', 'price' => 29.99],
    ['id' => 2, 'name' => 'Product 2', 'price' => 49.99]
];

foreach($products as $product) {
    echo "<div class='product'>";
    echo "<h3>{$product['name']}</h3>";
    echo "<p>\${$product['price']}</p>";
    echo "</div>";
}
?>`;
            
            virtualFS['cart.php'] = `<?php
session_start();
$_SESSION['cart'] = $_SESSION['cart'] ?? [];
?>`;
            
            await sleep(300);
            await addTerminalLine('✅ E-commerce platform created!', 'text-purple-400');
            updateFileTree();
        }

        async function createNodeAPI() {
            await addTerminalLine('🚀 Setting up Node.js REST API...', 'text-yellow-400');
            await sleep(500);
            
            virtualFS['server.js'] = `const express = require('express');
const app = express();
const port = 3000;

app.use(express.json());

app.get('/api/users', (req, res) => {
    res.json([
        { id: 1, name: 'User 1' },
        { id: 2, name: 'User 2' }
    ]);
});

app.listen(port, () => {
    console.log(\`API running on port \${port}\`);
});`;
            
            virtualFS['package.json'] = `{
  "name": "rest-api",
  "version": "1.0.0",
  "dependencies": {
    "express": "^4.18.0"
  }
}`;
            
            virtualFS['routes.js'] = `module.exports = {
    users: '/api/users',
    products: '/api/products'
};`;
            
            await sleep(300);
            await addTerminalLine('✅ REST API configured!', 'text-yellow-400');
            updateFileTree();
        }

        async function showStatus() {
            await addTerminalLine('Can Algorithm Status:', 'text-cyan-400');
            await addTerminalLine('━━━━━━━━━━━━━━━━━━━━━━━━', 'text-gray-600');
            await addTerminalLine('Version: 0.1.0 (Playground)', 'text-gray-400');
            await addTerminalLine('Mode: Demo/Simulation', 'text-gray-400');
            await addTerminalLine('Files: ' + Object.keys(virtualFS).length, 'text-gray-400');
            await addTerminalLine('Memory: Browser Storage', 'text-gray-400');
            await addTerminalLine('Status: Ready', 'text-green-400');
        }

        async function scanProject() {
            await addTerminalLine('🔍 Scanning project...', 'text-cyan-400');
            await sleep(500);
            
            if (Object.keys(virtualFS).length === 0) {
                await addTerminalLine('No files found. Create a project first.', 'text-yellow-400');
            } else {
                await addTerminalLine('Files found: ' + Object.keys(virtualFS).length, 'text-gray-400');
                for (let file in virtualFS) {
                    await addTerminalLine('  • ' + file, 'text-gray-500');
                    await sleep(100);
                }
                await addTerminalLine('✅ Scan complete', 'text-green-400');
            }
        }

        function updateFileTree() {
            const tree = document.getElementById('fileTree');
            if (Object.keys(virtualFS).length === 0) {
                tree.innerHTML = '<div class="text-gray-600 text-xs">No files yet</div>';
            } else {
                tree.innerHTML = '';
                for (let filename in virtualFS) {
                    const fileDiv = document.createElement('div');
                    fileDiv.className = 'py-1 px-2 hover:bg-gray-900 cursor-pointer rounded text-gray-300';
                    fileDiv.innerHTML = `<span class="file-icon"></span>${filename}`;
                    fileDiv.onclick = () => viewFile(filename);
                    tree.appendChild(fileDiv);
                }
            }
        }

        function viewFile(filename) {
            const viewer = document.getElementById('fileViewer');
            const content = document.getElementById('fileContent');
            viewer.classList.remove('hidden');
            content.textContent = virtualFS[filename];
        }

        async function addTerminalLine(text, className = 'text-gray-400') {
            const terminal = document.getElementById('terminalOutput');
            const line = document.createElement('div');
            line.className = className;
            line.textContent = text;
            terminal.insertBefore(line, terminal.lastElementChild);
            terminal.scrollTop = terminal.scrollHeight;
            await sleep(100);
        }

        function sleep(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }
    </script>
</body>
</html>