<?php
header('Content-Type: application/json');
require_once 'db.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Méthode non autorisée'
    ]);
    exit;
}

try {
    $db = getDB2Connection();
    
    // Récupérer les données du formulaire
    $type = $_POST['transaction-type'] ?? '';
    $amount = $_POST['amount'] ?? 0;
    $category = $_POST['category'] ?? '';
    $date = $_POST['date'] ?? '';
    $description = $_POST['description'] ?? '';
    
    // Valider les données
    if (empty($type) || empty($amount) || empty($category) || empty($date)) {
        echo json_encode([
            'success' => false,
            'message' => 'Tous les champs obligatoires doivent être remplis'
        ]);
        exit;
    }
    
    // Valider le type
    if ($type !== 'revenue' && $type !== 'expense') {
        echo json_encode([
            'success' => false,
            'message' => 'Type de transaction invalide'
        ]);
        exit;
    }
    
    // Valider le montant
    $amount = (float)$amount;
    if ($amount <= 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Le montant doit être supérieur à zéro'
        ]);
        exit;
    }
    
    // Insérer la transaction
    $stmt = $db->prepare("
        INSERT INTO finance_transactions (type, amount, category, date, description)
        VALUES (:type, :amount, :category, :date, :description)
    ");
    
    $stmt->bindParam(':type', $type);
    $stmt->bindParam(':amount', $amount);
    $stmt->bindParam(':category', $category);
    $stmt->bindParam(':date', $date);
    $stmt->bindParam(':description', $description);
    
    $stmt->execute();
    
    // Mettre à jour les objectifs liés
    updateGoalsProgress($db, $type);
    
    echo json_encode([
        'success' => true,
        'message' => 'Transaction ajoutée avec succès',
        'transaction_id' => $db->lastInsertId()
    ]);
    
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Erreur de base de données: ' . $e->getMessage()
    ]);
}

// Fonction pour mettre à jour les objectifs financiers
function updateGoalsProgress($db, $transactionType) {
    try {
        // Récupérer les objectifs actifs du même type
        $stmt = $db->prepare("
            SELECT id, type, target, current, end_date
            FROM finance_goals
            WHERE type = :type AND completed = 0 AND end_date >= CURDATE()
        ");
        
        // Pour les revenus, mettre à jour les objectifs de revenus et de bénéfices
        // Pour les dépenses, mettre à jour les objectifs de dépenses et de bénéfices
        if ($transactionType === 'revenue') {
            $goalType = 'revenue';
        } else {
            $goalType = 'expense';
        }
        
        $stmt->bindParam(':type', $goalType);
        $stmt->execute();
        $goals = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Pour chaque objectif, calculer le montant actuel
        foreach ($goals as $goal) {
            $startDate = date('Y-m-d', strtotime('-30 days'));
            $endDate = $goal['end_date'];
            
            // Requête pour calculer le total
            $calcStmt = $db->prepare("
                SELECT SUM(amount) as total
                FROM finance_transactions
                WHERE type = :type AND date BETWEEN :start AND :end
            ");
            
            $calcStmt->bindParam(':type', $transactionType);
            $calcStmt->bindParam(':start', $startDate);
            $calcStmt->bindParam(':end', $endDate);
            $calcStmt->execute();
            
            $result = $calcStmt->fetch(PDO::FETCH_ASSOC);
            $currentAmount = (float)($result['total'] ?? 0);
            
            // Mettre à jour l'objectif
            $updateStmt = $db->prepare("
                UPDATE finance_goals
                SET current = :current, completed = :completed
                WHERE id = :id
            ");
            
            $updateStmt->bindParam(':current', $currentAmount);
            $completed = ($currentAmount >= $goal['target']) ? 1 : 0;
            $updateStmt->bindParam(':completed', $completed);
            $updateStmt->bindParam(':id', $goal['id']);
            $updateStmt->execute();
        }
    } catch (PDOException $e) {
        // Gérer l'erreur silencieusement
        error_log('Erreur lors de la mise à jour des objectifs: ' . $e->getMessage());
    }
}
?>