document.addEventListener('DOMContentLoaded', () => {
    // Variables globales
    let financeChart = null;
    let financeData = {
        revenues: [],
        expenses: [],
        profit: [],
        dates: []
    };
    let goals = [];
    let transactions = [];

    // Éléments DOM
    const chartPeriodSelect = document.getElementById('chart-period');
    const chartTypeSelect = document.getElementById('chart-type');
    const addTransactionBtn = document.getElementById('add-transaction-btn');
    const transactionModal = document.getElementById('transaction-modal');
    const closeTransactionBtns = document.querySelectorAll('.close-transaction-modal');
    const transactionForm = document.getElementById('transaction-form');
    const transactionTypeRadios = document.querySelectorAll('input[name="transaction-type"]');
    const addGoalBtn = document.getElementById('add-goal-btn');
    const goalModal = document.getElementById('goal-modal');
    const closeGoalBtns = document.querySelectorAll('.close-goal-modal');
    const goalForm = document.getElementById('goal-form');

    // Initialiser les dates par défaut dans les formulaires
    const today = new Date();
    const formattedDate = today.toISOString().split('T')[0]; // Format YYYY-MM-DD
    
    if (document.getElementById('transaction-date')) {
        document.getElementById('transaction-date').value = formattedDate;
    }
    
    if (document.getElementById('goal-start-date')) {
        document.getElementById('goal-start-date').value = formattedDate;
        
        // Date d'échéance par défaut = 1 mois plus tard
        const nextMonth = new Date();
        nextMonth.setMonth(nextMonth.getMonth() + 1);
        document.getElementById('goal-end-date').value = nextMonth.toISOString().split('T')[0];
    }

    // Initialisation - Charger les données au démarrage
    initFinanceModule();

    function initFinanceModule() {
        // Initialiser le module financier
        const financeTab = document.querySelector('.finance-tab');
        if (financeTab) {
            financeTab.addEventListener('click', () => {
                loadFinanceData(30); // Charger les 30 derniers jours par défaut
                loadFinancialGoals();
                loadTransactions();
            });
        }

        // Charger les données au chargement si l'onglet finance est déjà actif
        if (window.location.hash === '#finance' || document.getElementById('finance-content')?.classList.contains('active')) {
            loadFinanceData(30);
            loadFinancialGoals();
            loadTransactions();
        }

        // Écouteurs d'événements pour les filtres du graphique
        if (chartPeriodSelect) {
            chartPeriodSelect.addEventListener('change', () => {
                loadFinanceData(parseInt(chartPeriodSelect.value));
            });
        }

        if (chartTypeSelect) {
            chartTypeSelect.addEventListener('change', () => {
                updateFinanceChart(financeData, chartTypeSelect.value);
            });
        }

        // Écouteurs d'événements pour le modal de transaction
        if (addTransactionBtn) {
            addTransactionBtn.addEventListener('click', () => {
                showTransactionModal();
            });
        }

        closeTransactionBtns.forEach(btn => {
            btn.addEventListener('click', () => {
                hideTransactionModal();
            });
        });

        if (transactionForm) {
            transactionForm.addEventListener('submit', handleTransactionSubmit);
        }

        // Basculer les catégories en fonction du type de transaction
        transactionTypeRadios.forEach(radio => {
            radio.addEventListener('change', toggleTransactionCategories);
        });

        // Écouteurs d'événements pour le modal d'objectif
        if (addGoalBtn) {
            addGoalBtn.addEventListener('click', () => {
                showGoalModal();
            });
        }

        closeGoalBtns.forEach(btn => {
            btn.addEventListener('click', () => {
                hideGoalModal();
            });
        });

        if (goalForm) {
            goalForm.addEventListener('submit', handleGoalSubmit);
        }

        // Gestionnaire pour le thème
        document.getElementById('theme-toggle')?.addEventListener('click', () => {
            setTimeout(updateChartTheme, 100);
        });
    }

    // Mise à jour du thème du graphique
    function updateChartTheme() {
        if (!financeChart) return;
        
        const isDarkMode = document.documentElement.classList.contains('dark');
        const textColor = isDarkMode ? '#E5E7EB' : '#4B5563';
        const gridColor = isDarkMode ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
        
        financeChart.options.scales.x.grid.color = gridColor;
        financeChart.options.scales.y.grid.color = gridColor;
        financeChart.options.scales.x.ticks.color = textColor;
        financeChart.options.scales.y.ticks.color = textColor;
        financeChart.update();
    }

    // Charger les données financières depuis la base de données
    async function loadFinanceData(days = 30) {
        try {
            showLoadingOverlay();
            
            const response = await fetch(`get_finance_data.php?days=${days}`);
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const data = await response.json();
            
            if (data.success) {
                financeData = data.data;
                updateFinanceSummary(data.data);
                updateFinanceChart(data.data, chartTypeSelect ? chartTypeSelect.value : 'line');
            } else {
                throw new Error(data.message || 'Erreur de chargement des données');
            }
            
            hideLoadingOverlay();
            return data.data;
        } catch (error) {
            hideLoadingOverlay();
            console.error('Erreur de chargement des données financières:', error);
            showNotification('Erreur de chargement des données financières', 'error');
        }
    }

    // Mettre à jour le résumé financier
    function updateFinanceSummary(data) {
        const totalRevenue = document.getElementById('total-revenue');
        const totalExpenses = document.getElementById('total-expenses');
        const totalProfit = document.getElementById('total-profit');
        const revenueTrend = document.getElementById('revenue-trend');
        const expensesTrend = document.getElementById('expenses-trend');
        const profitRatio = document.getElementById('profit-ratio');
        
        if (!totalRevenue || !totalExpenses || !totalProfit) return;
        
        const sumRevenue = data.revenues.reduce((sum, val) => sum + val, 0);
        const sumExpenses = data.expenses.reduce((sum, val) => sum + val, 0);
        const sumProfit = sumRevenue - sumExpenses;
        
        totalRevenue.textContent = `${sumRevenue.toLocaleString()} €`;
        totalExpenses.textContent = `${sumExpenses.toLocaleString()} €`;
        totalProfit.textContent = `${sumProfit.toLocaleString()} €`;
        
        // Calcul des tendances
        const revenueChange = calculateTrend(data.revenues);
        const expensesChange = calculateTrend(data.expenses);
        const profitPercentage = (sumRevenue > 0) ? ((sumProfit / sumRevenue) * 100).toFixed(1) : 0;
        
        revenueTrend.innerHTML = getTrendHTML(revenueChange, true);
        expensesTrend.innerHTML = getTrendHTML(expensesChange, false);
        profitRatio.textContent = `Marge bénéficiaire: ${profitPercentage}%`;
        
        // Mise à jour de la couleur du profit selon qu'il est positif ou négatif
        if (sumProfit >= 0) {
            totalProfit.classList.remove('text-red-500');
            totalProfit.classList.add('text-green-500');
        } else {
            totalProfit.classList.remove('text-green-500');
            totalProfit.classList.add('text-red-500');
        }
    }
    
    // Calculer la tendance (pourcentage d'évolution)
    function calculateTrend(data) {
        if (data.length < 2) return 0;
        
        const half = Math.floor(data.length / 2);
        const firstHalf = data.slice(0, half);
        const secondHalf = data.slice(half);
        
        const firstHalfAvg = firstHalf.reduce((sum, val) => sum + val, 0) / firstHalf.length;
        const secondHalfAvg = secondHalf.reduce((sum, val) => sum + val, 0) / secondHalf.length;
        
        if (firstHalfAvg === 0) return 0;
        return ((secondHalfAvg - firstHalfAvg) / firstHalfAvg) * 100;
    }
    
    // Générer le HTML pour afficher la tendance
    function getTrendHTML(percentage, isRevenue) {
        const fixed = percentage.toFixed(1);
        const isPositive = percentage > 0;
        
        // Pour les revenus, positif = bon ; pour les dépenses, négatif = bon
        const isGood = isRevenue ? isPositive : !isPositive;
        
        const color = isGood ? 'text-green-500' : 'text-red-500';
        const icon = isPositive ? 'fa-arrow-up' : 'fa-arrow-down';
        
        return `<span class="${color}"><i class="fas ${icon} mr-1"></i>${Math.abs(fixed)}%</span> depuis le mois dernier`;
    }

    // Initialiser et mettre à jour le graphique financier
    function updateFinanceChart(data, chartType = 'line') {
        const chartContainer = document.getElementById('finance-chart');
        if (!chartContainer) return;
        
        // Détruire le graphique existant s'il existe
        if (financeChart) {
            financeChart.destroy();
        }
        
        // Couleurs et options en fonction du thème
        const isDarkMode = document.documentElement.classList.contains('dark');
        const textColor = isDarkMode ? '#E5E7EB' : '#4B5563';
        const gridColor = isDarkMode ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
        
        // Configurer le graphique
        const ctx = chartContainer.getContext('2d');
        
        // Options spécifiques selon le type de graphique
        let config;
        
        if (chartType === 'line') {
            config = {
                type: 'line',
                data: {
                    labels: data.dates,
                    datasets: [
                        {
                            label: 'Revenus',
                            data: data.revenues,
                            borderColor: '#10B981',
                            backgroundColor: 'rgba(16, 185, 129, 0.1)',
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Dépenses',
                            data: data.expenses,
                            borderColor: '#EF4444',
                            backgroundColor: 'rgba(239, 68, 68, 0.1)',
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Bénéfice',
                            data: data.profit,
                            borderColor: '#3B82F6',
                            backgroundColor: 'rgba(59, 130, 246, 0.1)',
                            tension: 0.4,
                            fill: true
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: {
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor
                            }
                        },
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor,
                                callback: (value) => `${value.toLocaleString()} €`
                            }
                        }
                    },
                    interaction: {
                        mode: 'index',
                        intersect: false
                    },
                    plugins: {
                        legend: {
                            labels: {
                                color: textColor
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: (context) => `${context.dataset.label}: ${context.raw.toLocaleString()} €`
                            }
                        }
                    }
                }
            };
        } else if (chartType === 'bar') {
            config = {
                type: 'bar',
                data: {
                    labels: data.dates,
                    datasets: [
                        {
                            label: 'Revenus',
                            data: data.revenues,
                            backgroundColor: '#10B981',
                        },
                        {
                            label: 'Dépenses',
                            data: data.expenses,
                            backgroundColor: '#EF4444',
                        },
                        {
                            label: 'Bénéfice',
                            data: data.profit,
                            backgroundColor: '#3B82F6',
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: {
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor
                            }
                        },
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor,
                                callback: (value) => `${value.toLocaleString()} €`
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            labels: {
                                color: textColor
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: (context) => `${context.dataset.label}: ${context.raw.toLocaleString()} €`
                            }
                        }
                    }
                }
            };
        } else if (chartType === 'stacked') {
            config = {
                type: 'bar',
                data: {
                    labels: data.dates,
                    datasets: [
                        {
                            label: 'Revenus',
                            data: data.revenues,
                            backgroundColor: '#10B981',
                        },
                        {
                            label: 'Dépenses',
                            data: data.expenses.map(val => -val), // Valeurs négatives pour les dépenses
                            backgroundColor: '#EF4444',
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: {
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor
                            }
                        },
                        y: {
                            stacked: false,
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor,
                                callback: (value) => `${Math.abs(value).toLocaleString()} €`
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            labels: {
                                color: textColor
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: (context) => `${context.dataset.label}: ${Math.abs(context.raw).toLocaleString()} €`
                            }
                        }
                    }
                }
            };
        }
        
        financeChart = new Chart(ctx, config);
    }

    // Charger les objectifs financiers
    async function loadFinancialGoals() {
        try {
            const response = await fetch('get_financial_goals.php');
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const data = await response.json();
            
            if (data.success) {
                goals = data.goals;
                updateGoalsDisplay(data.goals);
                updateGoalSummary(data.goals);
            } else {
                throw new Error(data.message || 'Erreur de chargement des objectifs');
            }
            
            return data.goals;
        } catch (error) {
            console.error('Erreur de chargement des objectifs:', error);
            showNotification('Erreur de chargement des objectifs financiers', 'error');
        }
    }
    
    // Mettre à jour l'affichage des objectifs financiers
    function updateGoalsDisplay(goalsData) {
        const goalsContainer = document.getElementById('financial-goals');
        if (!goalsContainer) return;
        
        goalsContainer.innerHTML = '';
        
        if (goalsData.length === 0) {
            goalsContainer.innerHTML = `
                <div class="text-center text-gray-500 py-4">
                    Aucun objectif financier défini. Cliquez sur Ajouter pour créer un objectif.
                </div>
            `;
            return;
        }
        
        goalsData.forEach(goal => {
            const progress = Math.min(100, Math.round((goal.current / goal.target) * 100));
            
            // Couleur basée sur le type d'objectif
            let colorClass = '';
            if (goal.type === 'revenue') colorClass = 'bg-green-500';
            else if (goal.type === 'expense') colorClass = 'bg-red-500';
            else if (goal.type === 'profit') colorClass = 'bg-blue-500';
            else if (goal.type === 'savings') colorClass = 'bg-purple-500';
            
            // Formatage des dates
            const startDate = new Date(goal.start_date);
            const endDate = new Date(goal.end_date);
            const dateRange = `${startDate.toLocaleDateString()} - ${endDate.toLocaleDateString()}`;
            
            const goalElement = document.createElement('div');
            goalElement.className = 'bg-white dark:bg-gray-700 rounded-lg shadow-sm p-3';
            goalElement.innerHTML = `
                <div class="flex justify-between items-start mb-2">
                    <div>
                        <h3 class="font-medium text-sm">${goal.title}</h3>
                        <span class="text-xs text-gray-500">${dateRange}</span>
                    </div>
                    <div class="flex space-x-2">
                        ${goal.completed ? 
                            `<span class="bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200 text-xs px-2 py-1 rounded-full">
                                <i class="fas fa-check-circle mr-1"></i>Complété
                            </span>` : ``
                        }
                        <div class="text-gray-500 cursor-pointer hover:text-gray-700 dark:hover:text-gray-300 edit-goal" data-goal-id="${goal.id}">
                            <i class="fas fa-edit"></i>
                        </div>
                    </div>
                </div>
                <div class="flex justify-between items-center text-xs mb-1">
                    <span>${goal.current.toLocaleString()} € / ${goal.target.toLocaleString()} €</span>
                    <span>${progress}%</span>
                </div>
                <div class="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-1.5">
                    <div class="${colorClass} h-1.5 rounded-full" style="width: ${progress}%"></div>
                </div>
            `;
            
            // Ajouter un gestionnaire d'événements pour l'édition
            const editBtn = goalElement.querySelector('.edit-goal');
            if (editBtn) {
                editBtn.addEventListener('click', () => {
                    editGoal(goal.id);
                });
            }
            
            goalsContainer.appendChild(goalElement);
        });
    }
    
    // Mettre à jour le résumé des objectifs
    function updateGoalSummary(goalsData) {
        const goalsCompleted = document.getElementById('goals-completed');
        const goalProgress = document.getElementById('goal-progress');
        
        if (!goalsCompleted || !goalProgress) return;
        
        const completedCount = goalsData.filter(g => g.completed).length;
        
        goalsCompleted.textContent = `${completedCount}/${goalsData.length}`;
        
        // Calculer le pourcentage d'achèvement moyen des objectifs non terminés
        const ongoingGoals = goalsData.filter(g => !g.completed);
        
        if (ongoingGoals.length > 0) {
            const avgProgress = ongoingGoals.reduce((sum, g) => {
                return sum + (g.current / g.target);
            }, 0) / ongoingGoals.length * 100;
            
            goalProgress.textContent = `${avgProgress.toFixed(1)}% d'achèvement moyen`;
        } else {
            goalProgress.textContent = 'Tous les objectifs sont atteints';
        }
    }
    
    // Charger les transactions
    async function loadTransactions(limit = 10) {
        try {
            const transactionsContainer = document.getElementById('latest-transactions');
            if (!transactionsContainer) return;
            
            transactionsContainer.innerHTML = `
                <div class="text-center text-gray-500 py-4">
                    Chargement des transactions...
                </div>
            `;
            
            const response = await fetch(`get_transactions.php?limit=${limit}`);
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const data = await response.json();
            
            if (data.success) {
                transactions = data.transactions;
                updateTransactionsDisplay(data.transactions);
            } else {
                throw new Error(data.message || 'Erreur de chargement des transactions');
            }
            
            return data.transactions;
        } catch (error) {
            console.error('Erreur de chargement des transactions:', error);
            
            const transactionsContainer = document.getElementById('latest-transactions');
            if (transactionsContainer) {
                transactionsContainer.innerHTML = `
                    <div class="text-center text-red-500 py-4">
                        Erreur lors du chargement des transactions
                    </div>
                `;
            }
        }
    }
    
    // Mettre à jour l'affichage des transactions
    function updateTransactionsDisplay(transactionsData) {
        const transactionsContainer = document.getElementById('latest-transactions');
        if (!transactionsContainer) return;
        
        transactionsContainer.innerHTML = '';
        
        if (transactionsData.length === 0) {
            transactionsContainer.innerHTML = `
                <div class="text-center text-gray-500 py-4">
                    Aucune transaction. Cliquez sur Ajouter pour enregistrer une transaction.
                </div>
            `;
            return;
        }
        
        transactionsData.forEach(transaction => {
            const isRevenue = transaction.type === 'revenue';
            const date = new Date(transaction.date);
            
            const transactionElement = document.createElement('div');
            transactionElement.className = 'flex justify-between items-center p-3 border-b border-gray-100 dark:border-gray-700';
            transactionElement.innerHTML = `
                <div>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center ${isRevenue ? 'bg-green-100 dark:bg-green-900' : 'bg-red-100 dark:bg-red-900'}">
                            <i class="fas ${isRevenue ? 'fa-arrow-up text-green-500' : 'fa-arrow-down text-red-500'}"></i>
                        </div>
                        <div>
                            <div class="font-medium text-sm">${getCategoryLabel(transaction.category)}</div>
                            <div class="text-xs text-gray-500">${date.toLocaleDateString()}</div>
                        </div>
                    </div>
                </div>
                <div class="flex items-center">
                    <span class="${isRevenue ? 'text-green-500' : 'text-red-500'} font-medium">
                        ${isRevenue ? '+' : '-'}${transaction.amount.toLocaleString()} €
                    </span>
                    <div class="text-gray-500 cursor-pointer hover:text-gray-700 dark:hover:text-gray-300 ml-2 edit-transaction" data-transaction-id="${transaction.id}">
                        <i class="fas fa-edit"></i>
                    </div>
                </div>
            `;
            
            // Ajouter un gestionnaire d'événements pour l'édition
            const editBtn = transactionElement.querySelector('.edit-transaction');
            if (editBtn) {
                editBtn.addEventListener('click', () => {
                    editTransaction(transaction.id);
                });
            }
            
            transactionsContainer.appendChild(transactionElement);
        });
    }
    
    // Obtenir le libellé d'une catégorie
    function getCategoryLabel(categoryId) {
        const categories = {
            // Revenus
            'sales': 'Ventes',
            'services': 'Services',
            'subscriptions': 'Abonnements',
            'investments': 'Investissements',
            'other_revenue': 'Autre revenu',
            // Dépenses
            'salaries': 'Salaires',
            'rent': 'Loyer',
            'equipment': 'Équipement',
            'marketing': 'Marketing',
            'software': 'Logiciels',
            'taxes': 'Taxes',
            'other_expense': 'Autre dépense'
        };
        
        return categories[categoryId] || categoryId;
    }
    
    // Basculer les catégories du formulaire de transaction
    function toggleTransactionCategories() {
        const type = document.querySelector('input[name="transaction-type"]:checked').value;
        const categorySelect = document.getElementById('transaction-category');
        
        if (!categorySelect) return;
        
        const revenueCategories = categorySelect.querySelector('#revenue-categories');
        const expenseCategories = categorySelect.querySelector('#expense-categories');
        
        if (type === 'revenue') {
            revenueCategories.disabled = false;
            expenseCategories.disabled = true;
            
            // Sélectionner la première option de revenu
            if (revenueCategories.querySelector('option')) {
                revenueCategories.querySelector('option').selected = true;
            }
        } else {
            revenueCategories.disabled = true;
            expenseCategories.disabled = false;
            
            // Sélectionner la première option de dépense
            if (expenseCategories.querySelector('option')) {
                expenseCategories.querySelector('option').selected = true;
            }
        }
    }
    
    // Afficher le modal de transaction
    function showTransactionModal(isEdit = false) {
        const modal = document.getElementById('transaction-modal');
        const title = document.getElementById('transaction-modal-title');
        
        if (!modal || !title) return;
        
        title.innerHTML = `
            <i class="fas fa-exchange-alt text-gray-600 dark:text-gray-400"></i>
            <span>${isEdit ? 'Modifier la transaction' : 'Nouvelle transaction'}</span>
        `;
        
        if (!isEdit) {
            // Réinitialiser le formulaire pour une nouvelle transaction
            document.getElementById('transaction-form').reset();
            document.getElementById('transaction-id').value = '';
            document.getElementById('transaction-date').value = new Date().toISOString().split('T')[0];
            
            // Sélectionner le type "revenu" par défaut
            const revenueRadio = document.querySelector('input[name="transaction-type"][value="revenue"]');
            if (revenueRadio) {
                revenueRadio.checked = true;
            }
            
            toggleTransactionCategories();
        }
        
        modal.classList.remove('hidden');
    }
    
    // Masquer le modal de transaction
    function hideTransactionModal() {
        const modal = document.getElementById('transaction-modal');
        if (modal) {
            modal.classList.add('hidden');
        }
    }
    
    // Éditer une transaction existante
    async function editTransaction(id) {
        try {
            showLoadingOverlay();
            
            const response = await fetch(`get_transaction.php?id=${id}`);
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const data = await response.json();
            
            if (data.success) {
                const transaction = data.transaction;
                
                // Remplir le formulaire avec les données
                document.getElementById('transaction-id').value = transaction.id;
                document.getElementById('transaction-amount').value = transaction.amount;
                document.getElementById('transaction-date').value = transaction.date;
                document.getElementById('transaction-description').value = transaction.description || '';
                
                // Sélectionner le type
                const typeRadio = document.querySelector(`input[name="transaction-type"][value="${transaction.type}"]`);
                if (typeRadio) {
                    typeRadio.checked = true;
                }
                
                toggleTransactionCategories();
                
                // Sélectionner la catégorie
                const categorySelect = document.getElementById('transaction-category');
                if (categorySelect) {
                    const categoryOption = categorySelect.querySelector(`option[value="${transaction.category}"]`);
                    if (categoryOption) {
                        categoryOption.selected = true;
                    }
                }
                
                showTransactionModal(true);
            } else {
                throw new Error(data.message || 'Erreur de chargement de la transaction');
            }
            
            hideLoadingOverlay();
        } catch (error) {
            hideLoadingOverlay();
            console.error('Erreur lors de l\'édition de la transaction:', error);
            showNotification('Erreur lors de l\'édition de la transaction', 'error');
        }
    }
    
    // Gérer la soumission du formulaire de transaction
    async function handleTransactionSubmit(e) {
        e.preventDefault();
        
        const form = e.target;
        const submitButton = form.querySelector('button[type="submit"]');
        const originalContent = submitButton.innerHTML;
        const isEdit = document.getElementById('transaction-id').value !== '';
        
        showLoadingOverlay();
        submitButton.disabled = true;
        submitButton.innerHTML = `
            <div class="flex items-center space-x-2">
                <div class="animate-spin rounded-full h-4 w-4 border-2 border-gray-500 border-t-white"></div>
                <span>${isEdit ? 'Enregistrement...' : 'Création...'}</span>
            </div>
        `;

        try {
            const formData = new FormData(form);
            const response = await fetch(isEdit ? 'update_transaction.php' : 'add_transaction.php', {
                method: 'POST',
                body: formData
            });
            
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const result = await response.json();
            
            if (result.success) {
                hideTransactionModal();
                showNotification(`Transaction ${isEdit ? 'mise à jour' : 'ajoutée'} avec succès`, 'success');
                
                // Recharger les données
                loadTransactions();
                loadFinanceData(parseInt(chartPeriodSelect?.value || 30));
            } else {
                throw new Error(result.message || `Échec de l'${isEdit ? 'édition' : 'ajout'} de la transaction`);
            }
        } catch (error) {
            console.error('Erreur:', error);
            showNotification(error.message, 'error');
        } finally {
            hideLoadingOverlay();
            submitButton.disabled = false;
            submitButton.innerHTML = originalContent;
        }
    }
    
    // Afficher le modal d'objectif
    function showGoalModal(isEdit = false) {
        const modal = document.getElementById('goal-modal');
        const title = document.getElementById('goal-modal-title');
        
        if (!modal || !title) return;
        
        title.innerHTML = `
            <i class="fas fa-bullseye text-gray-600 dark:text-gray-400"></i>
            <span>${isEdit ? 'Modifier l\'objectif' : 'Nouvel objectif'}</span>
        `;
        
        if (!isEdit) {
            // Réinitialiser le formulaire pour un nouvel objectif
            document.getElementById('goal-form').reset();
            document.getElementById('goal-id').value = '';
            
            // Dates par défaut
            const today = new Date().toISOString().split('T')[0];
            const nextMonth = new Date();
            nextMonth.setMonth(nextMonth.getMonth() + 1);
            
            document.getElementById('goal-start-date').value = today;
            document.getElementById('goal-end-date').value = nextMonth.toISOString().split('T')[0];
        }
        
        modal.classList.remove('hidden');
    }
    
    // Masquer le modal d'objectif
    function hideGoalModal() {
        const modal = document.getElementById('goal-modal');
        if (modal) {
            modal.classList.add('hidden');
        }
    }
    
    // Éditer un objectif existant
    async function editGoal(id) {
        try {
            showLoadingOverlay();
            
            const response = await fetch(`get_goal.php?id=${id}`);
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const data = await response.json();
            
            if (data.success) {
                const goal = data.goal;
                
                // Remplir le formulaire avec les données
                document.getElementById('goal-id').value = goal.id;
                document.getElementById('goal-title').value = goal.title;
                document.getElementById('goal-type').value = goal.type;
                document.getElementById('goal-target').value = goal.target;
                document.getElementById('goal-start-date').value = goal.start_date;
                document.getElementById('goal-end-date').value = goal.end_date;
                document.getElementById('goal-notes').value = goal.notes || '';
                
                showGoalModal(true);
            } else {
                throw new Error(data.message || 'Erreur de chargement de l\'objectif');
            }
            
            hideLoadingOverlay();
        } catch (error) {
            hideLoadingOverlay();
            console.error('Erreur lors de l\'édition de l\'objectif:', error);
            showNotification('Erreur lors de l\'édition de l\'objectif', 'error');
        }
    }
    
    // Gérer la soumission du formulaire d'objectif
    async function handleGoalSubmit(e) {
        e.preventDefault();
        
        const form = e.target;
        const submitButton = form.querySelector('button[type="submit"]');
        const originalContent = submitButton.innerHTML;
        const isEdit = document.getElementById('goal-id').value !== '';
        
        showLoadingOverlay();
        submitButton.disabled = true;
        submitButton.innerHTML = `
            <div class="flex items-center space-x-2">
                <div class="animate-spin rounded-full h-4 w-4 border-2 border-gray-500 border-t-white"></div>
                <span>${isEdit ? 'Enregistrement...' : 'Création...'}</span>
            </div>
        `;

        try {
            const formData = new FormData(form);
            const response = await fetch(isEdit ? 'update_goal.php' : 'add_goal.php', {
                method: 'POST',
                body: formData
            });
            
            if (!response.ok) {
                throw new Error('Erreur serveur');
            }
            
            const result = await response.json();
            
            if (result.success) {
                hideGoalModal();
                showNotification(`Objectif ${isEdit ? 'mis à jour' : 'ajouté'} avec succès`, 'success');
                
                // Recharger les objectifs
                loadFinancialGoals();
            } else {
                throw new Error(result.message || `Échec de l'${isEdit ? 'édition' : 'ajout'} de l'objectif`);
            }
        } catch (error) {
            console.error('Erreur:', error);
            showNotification(error.message, 'error');
        } finally {
            hideLoadingOverlay();
            submitButton.disabled = false;
            submitButton.innerHTML = originalContent;
        }
    }
    
    // Fonction d'affichage des notifications déjà définie dans votre dashboard.js
    function showNotification(message, type = 'success') {
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-lg z-50 ${
            type === 'success' ? 'bg-green-500' : 'bg-red-500'
        } text-white`;
        
        notification.innerHTML = `
            <div class="flex items-center space-x-2">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'}"></i>
                <span>${message}</span>
            </div>
        `;
        
        document.body.appendChild(notification);
        setTimeout(() => {
            notification.remove();
        }, 3000);
    }
    
    // Fonctions pour afficher/masquer l'overlay de chargement
    function showLoadingOverlay() {
        const overlay = document.querySelector('.loading-overlay');
        if (overlay) {
            overlay.classList.remove('hidden');
        }
    }
    
    function hideLoadingOverlay() {
        const overlay = document.querySelector('.loading-overlay');
        if (overlay) {
            overlay.classList.add('hidden');
        }
    }
});