<?php

class ApiConnector {
    
    private function getConfig($resource) {
        $config_file = __DIR__ . '/config/' . $resource . '.json';
        
        if (!file_exists($config_file)) {
            throw new Exception("API config not found for: " . $resource);
        }
        
        return json_decode(file_get_contents($config_file), true);
    }
    
    private function makeRequest($method, $url, $data = null, $headers = []) {
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        switch (strtoupper($method)) {
            case 'POST':
                curl_setopt($ch, CURLOPT_POST, true);
                if ($data) {
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                }
                break;
            case 'PUT':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                if ($data) {
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                }
                break;
            case 'DELETE':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
                break;
        }
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_error($ch)) {
            throw new Exception("cURL Error: " . curl_error($ch));
        }
        
        curl_close($ch);
        
        if ($http_code >= 400) {
            throw new Exception("HTTP Error: " . $http_code);
        }
        
        return json_decode($response, true) ?: $response;
    }
    
    public function get($params, $resource) {
        $config = $this->getConfig($resource);
        
        if (preg_match("/(\w+)\s+where\s+(.+)/i", $params, $matches)) {
            $endpoint = $matches[1];
            $query_params = $this->parseQueryParams($matches[2]);
            
            $url = $config['base_url'] . '/' . $endpoint;
            if (!empty($query_params)) {
                $url .= '?' . http_build_query($query_params);
            }
            
            $headers = $this->buildHeaders($config);
            
            return $this->makeRequest('GET', $url, null, $headers);
        }
        
        throw new Exception("Invalid get syntax for API");
    }
    
    public function send($params, $resource) {
        $config = $this->getConfig($resource);
        
        if (preg_match("/data\s+['\"]({.*?})['\"]\s+to\s+endpoint\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $data = $matches[1];
            $endpoint = $matches[2];
            
            $url = $config['base_url'] . '/' . $endpoint;
            $headers = $this->buildHeaders($config);
            $headers[] = 'Content-Type: application/json';
            
            return $this->makeRequest('POST', $url, $data, $headers);
        }
        
        throw new Exception("Invalid send syntax for API");
    }
    
    public function search($params, $resource) {
        $config = $this->getConfig($resource);
        
        if (preg_match("/(\w+)\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $search_type = $matches[1];
            $search_term = $matches[2];
            
            $endpoint = $config['search_endpoint'] ?? 'search';
            $url = $config['base_url'] . '/' . $endpoint;
            
            $query_params = [
                $search_type => $search_term
            ];
            
            $url .= '?' . http_build_query($query_params);
            $headers = $this->buildHeaders($config);
            
            return $this->makeRequest('GET', $url, null, $headers);
        }
        
        throw new Exception("Invalid search syntax for API");
    }
    
    public function update($params, $resource) {
        $config = $this->getConfig($resource);
        
        if (preg_match("/(\w+)\s+id=(\d+)\s+with\s+['\"]({.*?})['\"]/i", $params, $matches)) {
            $endpoint = $matches[1];
            $id = $matches[2];
            $data = $matches[3];
            
            $url = $config['base_url'] . '/' . $endpoint . '/' . $id;
            $headers = $this->buildHeaders($config);
            $headers[] = 'Content-Type: application/json';
            
            return $this->makeRequest('PUT', $url, $data, $headers);
        }
        
        throw new Exception("Invalid update syntax for API");
    }
    
    public function delete($params, $resource) {
        $config = $this->getConfig($resource);
        
        if (preg_match("/(\w+)\s+id=(\d+)/i", $params, $matches)) {
            $endpoint = $matches[1];
            $id = $matches[2];
            
            $url = $config['base_url'] . '/' . $endpoint . '/' . $id;
            $headers = $this->buildHeaders($config);
            
            return $this->makeRequest('DELETE', $url, null, $headers);
        }
        
        throw new Exception("Invalid delete syntax for API");
    }
    
    private function parseQueryParams($query_string) {
        $params = [];
        
        if (preg_match_all("/(\w+)=['\"]?([^'\"&]+)['\"]?/", $query_string, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $params[$match[1]] = $match[2];
            }
        }
        
        return $params;
    }
    
    private function buildHeaders($config) {
        $headers = [];
        
        if (isset($config['auth_type'])) {
            switch ($config['auth_type']) {
                case 'bearer':
                    $headers[] = 'Authorization: Bearer ' . $config['token'];
                    break;
                case 'api_key':
                    $headers[] = $config['api_key_header'] . ': ' . $config['api_key'];
                    break;
                case 'basic':
                    $auth = base64_encode($config['username'] . ':' . $config['password']);
                    $headers[] = 'Authorization: Basic ' . $auth;
                    break;
            }
        }
        
        if (isset($config['custom_headers'])) {
            $headers = array_merge($headers, $config['custom_headers']);
        }
        
        return $headers;
    }
}

?>