(function() {
    const WEBSOCKET_URL = 'wss://aktascorp.com:8081';
    const CHATBOX_ID = 'imators-chatbox';
    
    let ws = null;
    let conversationId = null;
    let visitorId = localStorage.getItem('imators_visitor_id') || generateId();
    localStorage.setItem('imators_visitor_id', visitorId);
    
    function generateId() {
        return 'visitor_' + Math.random().toString(36).substr(2, 9) + Date.now();
    }
    
    function injectStyles() {
        const style = document.createElement('style');
        style.textContent = `
            #${CHATBOX_ID} {
                position: fixed;
                bottom: 20px;
                right: 20px;
                width: 380px;
                height: 600px;
                z-index: 999999;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            }
            
            #${CHATBOX_ID}.minimized {
                height: 60px;
                width: 60px;
            }
            
            #${CHATBOX_ID} .chat-toggle {
                position: absolute;
                bottom: 0;
                right: 0;
                width: 60px;
                height: 60px;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                border-radius: 30px;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                box-shadow: 0 4px 20px rgba(0,0,0,0.2);
                transition: all 0.3s;
            }
            
            #${CHATBOX_ID} .chat-toggle:hover {
                transform: scale(1.1);
                box-shadow: 0 6px 30px rgba(0,0,0,0.3);
            }
            
            #${CHATBOX_ID} .chat-toggle svg {
                width: 30px;
                height: 30px;
                fill: white;
                transition: transform 0.3s;
            }
            
            #${CHATBOX_ID}.minimized .chat-toggle svg {
                transform: rotate(0deg);
            }
            
            #${CHATBOX_ID}:not(.minimized) .chat-toggle svg {
                transform: rotate(45deg);
            }
            
            #${CHATBOX_ID} .chat-window {
                position: absolute;
                bottom: 80px;
                right: 0;
                width: 100%;
                height: calc(100% - 80px);
                background: rgba(255, 255, 255, 0.95);
                backdrop-filter: blur(20px);
                border-radius: 20px;
                box-shadow: 0 20px 60px rgba(0,0,0,0.15);
                display: flex;
                flex-direction: column;
                opacity: 1;
                transform: translateY(0);
                transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
                overflow: hidden;
            }
            
            #${CHATBOX_ID}.minimized .chat-window {
                opacity: 0;
                transform: translateY(20px);
                pointer-events: none;
            }
            
            #${CHATBOX_ID} .chat-header {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 20px;
                border-radius: 20px 20px 0 0;
                flex-shrink: 0;
            }
            
            #${CHATBOX_ID} .chat-header h3 {
                margin: 0;
                font-size: 18px;
                font-weight: 600;
            }
            
            #${CHATBOX_ID} .chat-header p {
                margin: 5px 0 0 0;
                font-size: 14px;
                opacity: 0.9;
            }
            
            #${CHATBOX_ID} .chat-messages {
                flex: 1;
                overflow-y: auto;
                padding: 20px;
                background: rgba(249, 250, 251, 0.8);
            }
            
            #${CHATBOX_ID} .message {
                margin-bottom: 15px;
                display: flex;
                align-items: flex-end;
                animation: messageSlide 0.3s ease-out;
            }
            
            @keyframes messageSlide {
                from {
                    opacity: 0;
                    transform: translateY(10px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            #${CHATBOX_ID} .message.visitor {
                justify-content: flex-end;
            }
            
            #${CHATBOX_ID} .message.admin {
                justify-content: flex-start;
            }
            
            #${CHATBOX_ID} .message-bubble {
                max-width: 70%;
                padding: 12px 18px;
                border-radius: 18px;
                font-size: 14px;
                line-height: 1.4;
                position: relative;
            }
            
            #${CHATBOX_ID} .message.visitor .message-bubble {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                border-bottom-right-radius: 4px;
            }
            
            #${CHATBOX_ID} .message.admin .message-bubble {
                background: white;
                color: #333;
                border-bottom-left-radius: 4px;
                box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            }
            
            #${CHATBOX_ID} .message-time {
                font-size: 11px;
                opacity: 0.6;
                margin-top: 4px;
            }
            
            #${CHATBOX_ID} .chat-input {
                padding: 20px;
                background: white;
                border-top: 1px solid rgba(0,0,0,0.05);
                display: flex;
                gap: 10px;
                align-items: center;
            }
            
            #${CHATBOX_ID} .chat-input input {
                flex: 1;
                padding: 12px 18px;
                border: 2px solid #e5e7eb;
                border-radius: 25px;
                font-size: 14px;
                outline: none;
                transition: all 0.2s;
            }
            
            #${CHATBOX_ID} .chat-input input:focus {
                border-color: #667eea;
            }
            
            #${CHATBOX_ID} .chat-input button {
                width: 44px;
                height: 44px;
                border-radius: 22px;
                border: none;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                transition: all 0.2s;
                flex-shrink: 0;
            }
            
            #${CHATBOX_ID} .chat-input button:hover {
                transform: scale(1.05);
                box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
            }
            
            #${CHATBOX_ID} .chat-input button:active {
                transform: scale(0.95);
            }
            
            #${CHATBOX_ID} .chat-input button svg {
                width: 20px;
                height: 20px;
                fill: white;
            }
            
            #${CHATBOX_ID} .connection-status {
                position: absolute;
                top: 10px;
                right: 10px;
                width: 8px;
                height: 8px;
                border-radius: 50%;
                background: #10b981;
                box-shadow: 0 0 0 3px rgba(16, 185, 129, 0.2);
            }
            
            #${CHATBOX_ID} .connection-status.disconnected {
                background: #ef4444;
                box-shadow: 0 0 0 3px rgba(239, 68, 68, 0.2);
            }
            
            #${CHATBOX_ID} .typing-indicator {
                display: none;
                padding: 20px;
                padding-top: 0;
            }
            
            #${CHATBOX_ID} .typing-indicator.active {
                display: block;
            }
            
            #${CHATBOX_ID} .typing-dots {
                display: inline-flex;
                gap: 4px;
            }
            
            #${CHATBOX_ID} .typing-dots span {
                width: 8px;
                height: 8px;
                border-radius: 50%;
                background: #9ca3af;
                animation: typingDot 1.4s infinite;
            }
            
            #${CHATBOX_ID} .typing-dots span:nth-child(2) {
                animation-delay: 0.2s;
            }
            
            #${CHATBOX_ID} .typing-dots span:nth-child(3) {
                animation-delay: 0.4s;
            }
            
            @keyframes typingDot {
                0%, 60%, 100% {
                    opacity: 0.3;
                    transform: scale(1);
                }
                30% {
                    opacity: 1;
                    transform: scale(1.2);
                }
            }
            
            @media (max-width: 480px) {
                #${CHATBOX_ID} {
                    width: 100%;
                    height: 100%;
                    bottom: 0;
                    right: 0;
                    max-width: none;
                }
                
                #${CHATBOX_ID} .chat-window {
                    bottom: 0;
                    height: 100%;
                    border-radius: 0;
                }
                
                #${CHATBOX_ID} .chat-header {
                    border-radius: 0;
                }
                
                #${CHATBOX_ID}.minimized {
                    width: 60px;
                    height: 60px;
                    bottom: 20px;
                    right: 20px;
                }
            }
        `;
        document.head.appendChild(style);
    }
    
    function createChatbox() {
        const chatbox = document.createElement('div');
        chatbox.id = CHATBOX_ID;
        chatbox.className = 'minimized';
        chatbox.innerHTML = `
            <div class="chat-window">
                <div class="connection-status"></div>
                <div class="chat-header">
                    <h3>Support en ligne</h3>
                    <p>Nous sommes là pour vous aider</p>
                </div>
                <div class="chat-messages"></div>
                <div class="typing-indicator">
                    <div class="typing-dots">
                        <span></span>
                        <span></span>
                        <span></span>
                    </div>
                </div>
                <div class="chat-input">
                    <input type="text" placeholder="Écrivez votre message..." />
                    <button type="button">
                        <svg viewBox="0 0 24 24">
                            <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                        </svg>
                    </button>
                </div>
            </div>
            <div class="chat-toggle">
                <svg viewBox="0 0 24 24">
                    <path d="M20 2H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h14l4 4V4c0-1.1-.9-2-2-2z"/>
                </svg>
            </div>
        `;
        
        document.body.appendChild(chatbox);
        
        const toggle = chatbox.querySelector('.chat-toggle');
        const input = chatbox.querySelector('.chat-input input');
        const sendButton = chatbox.querySelector('.chat-input button');
        const messagesContainer = chatbox.querySelector('.chat-messages');
        
        toggle.addEventListener('click', () => {
            chatbox.classList.toggle('minimized');
            if (!chatbox.classList.contains('minimized')) {
                input.focus();
                messagesContainer.scrollTop = messagesContainer.scrollHeight;
            }
        });
        
        function sendMessage() {
            const message = input.value.trim();
            if (message && ws && ws.readyState === WebSocket.OPEN) {
                ws.send(JSON.stringify({
                    type: 'message',
                    message: message
                }));
                
                addMessage('visitor', message);
                input.value = '';
                input.focus();
            }
        }
        
        sendButton.addEventListener('click', sendMessage);
        input.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                sendMessage();
            }
        });
    }
    
    function addMessage(sender, message, messageId) {
        const messagesContainer = document.querySelector(`#${CHATBOX_ID} .chat-messages`);
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}`;
        
        const time = new Date().toLocaleTimeString('fr-FR', { 
            hour: '2-digit', 
            minute: '2-digit' 
        });
        
        messageDiv.innerHTML = `
            <div class="message-bubble">
                ${message}
                <div class="message-time">${time}</div>
            </div>
        `;
        
        messagesContainer.appendChild(messageDiv);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
    
    function connectWebSocket() {
        ws = new WebSocket(WEBSOCKET_URL);
        const statusIndicator = document.querySelector(`#${CHATBOX_ID} .connection-status`);
        
        ws.onopen = () => {
            statusIndicator.classList.remove('disconnected');
            ws.send(JSON.stringify({
                type: 'init',
                domain: window.location.hostname,
                visitorId: visitorId
            }));
        };
        
        ws.onmessage = (event) => {
            const data = JSON.parse(event.data);
            
            switch(data.type) {
                case 'init_complete':
                    conversationId = data.conversationId;
                    const messagesContainer = document.querySelector(`#${CHATBOX_ID} .chat-messages`);
                    messagesContainer.innerHTML = '';
                    data.messages.forEach(msg => {
                        addMessage(msg.sender, msg.message, msg.id);
                    });
                    break;
                    
                case 'admin_message':
                    addMessage('admin', data.message, data.messageId);
                    showNotification('Nouveau message', data.message);
                    break;
            }
        };
        
        ws.onclose = () => {
            statusIndicator.classList.add('disconnected');
            setTimeout(connectWebSocket, 3000);
        };
        
        ws.onerror = () => {
            statusIndicator.classList.add('disconnected');
        };
    }
    
    function showNotification(title, body) {
        if ('Notification' in window && Notification.permission === 'granted') {
            new Notification(title, { body: body, icon: '/favicon.ico' });
        }
    }
    
    function init() {
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', init);
            return;
        }
        
        injectStyles();
        createChatbox();
        connectWebSocket();
        
        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission();
        }
    }
    
    init();
})();