<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>ISE Engine</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Urbanist:wght@100;200;300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://use.fontawesome.com/releases/v6.4.0/css/all.css">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        body { 
            font-family: 'Urbanist', sans-serif; 
            background: #fafafa;
        }
        .editor-area { 
            min-height: 80vh; 
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .toolbar-btn { 
            width: 36px; 
            height: 36px; 
            border-radius: 8px; 
            border: none; 
            background: transparent; 
            transition: all 0.2s;
        }
        .toolbar-btn:hover { 
            background: #f3f4f6; 
        }
        .toolbar-btn.active { 
            background: #3b82f6; 
            color: white; 
        }
        .ai-sidebar {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .ai-btn {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 12px;
            transition: all 0.2s;
        }
        .ai-btn:hover {
            background: #f1f5f9;
            border-color: #cbd5e1;
        }
        .editor-content:empty:before {
            content: attr(data-placeholder);
            color: #9ca3af;
        }
        .floating-toolbar {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            padding: 12px;
            display: flex;
            gap: 8px;
            z-index: 50;
        }
        .save-btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s;
        }
        .save-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }
        .modal-backdrop {
            background: rgba(0,0,0,0.6);
            backdrop-filter: blur(4px);
        }
    </style>
</head>
<body>
    <!-- Floating Toolbar -->
    <div class="floating-toolbar">
        <button id="save-btn" class="save-btn">
            <i class="fas fa-save mr-2"></i>Save
        </button>
        <button id="share-btn" class="save-btn" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
            <i class="fas fa-share mr-2"></i>Share
        </button>
        <a href="/workspace" class="save-btn" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
            <i class="fas fa-arrow-left mr-2"></i>Back
        </a>
    </div>

    <!-- Main Container -->
    <div class="min-h-screen p-6">
        <div class="max-w-7xl mx-auto">
            <div class="grid grid-cols-12 gap-6">
                
                <!-- Editor Section -->
                <div class="col-span-8">
                    <div class="editor-area p-8">
                        <!-- Title -->
                        <input type="text" id="document-title-input" 
                               placeholder="Untitled Document" 
                               class="w-full text-4xl font-light border-none outline-none mb-8 text-gray-800 placeholder-gray-400">
                        
                        <!-- Toolbar -->
                        <div class="flex items-center gap-2 mb-6 pb-4 border-b border-gray-100">
                            <button class="toolbar-btn" data-action="bold">
                                <i class="fas fa-bold"></i>
                            </button>
                            <button class="toolbar-btn" data-action="italic">
                                <i class="fas fa-italic"></i>
                            </button>
                            <button class="toolbar-btn" data-action="underline">
                                <i class="fas fa-underline"></i>
                            </button>
                            <div class="w-px h-6 bg-gray-200 mx-2"></div>
                            <button class="toolbar-btn" data-action="h1">H1</button>
                            <button class="toolbar-btn" data-action="h2">H2</button>
                            <button class="toolbar-btn" data-action="h3">H3</button>
                            <div class="w-px h-6 bg-gray-200 mx-2"></div>
                            <button class="toolbar-btn" data-action="bullet-list">
                                <i class="fas fa-list-ul"></i>
                            </button>
                            <button class="toolbar-btn" data-action="number-list">
                                <i class="fas fa-list-ol"></i>
                            </button>
                        </div>
                        
                        <!-- Editor -->
                        <div id="editor" class="editor-content outline-none text-gray-800 leading-relaxed text-lg" 
                             contenteditable="true" 
                             data-placeholder="Start writing...">
                        </div>
                    </div>
                </div>

                <!-- AI Sidebar -->
                <div class="col-span-4">
                    <div class="ai-sidebar p-6 sticky top-6">
                        <!-- Header -->
                        <div class="flex items-center mb-6">
                            <div class="w-10 h-10 bg-gradient-to-r from-purple-400 to-pink-400 rounded-full flex items-center justify-center mr-3">
                                <i class="fas fa-brain text-white"></i>
                            </div>
                            <h2 class="text-xl font-semibold text-gray-800">AI Assistant</h2>
                        </div>

                        <!-- Tabs -->
                        <div class="flex mb-6">
                            <button id="tab-claude" class="flex-1 py-3 px-4 bg-blue-500 text-white rounded-l-lg text-sm font-medium">
                                Claude
                            </button>
                            <button id="tab-perplexity" class="flex-1 py-3 px-4 bg-gray-100 text-gray-600 rounded-r-lg text-sm font-medium">
                                Search
                            </button>
                        </div>

                        <!-- Claude Panel -->
                        <div id="claude-panel" class="space-y-4">
                            <!-- Quick Actions -->
                            <div class="grid grid-cols-2 gap-3">
                                <button data-claude-action="grammar" class="ai-btn text-left">
                                    <div class="text-green-500 text-xl mb-2">
                                        <i class="fas fa-spell-check"></i>
                                    </div>
                                    <div class="text-sm font-medium">Fix Grammar</div>
                                </button>
                                <button data-claude-action="expand" class="ai-btn text-left">
                                    <div class="text-blue-500 text-xl mb-2">
                                        <i class="fas fa-expand-arrows-alt"></i>
                                    </div>
                                    <div class="text-sm font-medium">Expand</div>
                                </button>
                                <button data-claude-action="summarize" class="ai-btn text-left">
                                    <div class="text-orange-500 text-xl mb-2">
                                        <i class="fas fa-compress-arrows-alt"></i>
                                    </div>
                                    <div class="text-sm font-medium">Summarize</div>
                                </button>
                                <button data-claude-action="rephrase" class="ai-btn text-left">
                                    <div class="text-purple-500 text-xl mb-2">
                                        <i class="fas fa-magic"></i>
                                    </div>
                                    <div class="text-sm font-medium">Rephrase</div>
                                </button>
                            </div>

                            <!-- Custom Prompt -->
                            <div class="mt-6">
                                <textarea id="claude-prompt" 
                                          placeholder="Ask Claude anything..." 
                                          class="w-full p-4 border border-gray-200 rounded-lg text-sm resize-none focus:ring-2 focus:ring-blue-500 focus:border-transparent" 
                                          rows="3"></textarea>
                                <button id="claude-submit" class="w-full mt-3 bg-purple-500 text-white py-3 rounded-lg text-sm font-medium hover:bg-purple-600 transition-colors">
                                    Send to Claude
                                </button>
                            </div>

                            <!-- Response -->
                            <div id="claude-response" class="hidden mt-4 p-4 bg-purple-50 rounded-lg">
                                <div class="flex items-center justify-between mb-3">
                                    <span class="font-medium text-purple-700">Claude Response</span>
                                    <button id="claude-close" class="text-gray-400 hover:text-gray-600">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                <div id="claude-content" class="text-gray-700 text-sm mb-3"></div>
                                <button id="claude-apply" class="w-full bg-purple-500 text-white py-2 rounded-lg text-sm hover:bg-purple-600">
                                    Apply Changes
                                </button>
                            </div>
                        </div>

                        <!-- Perplexity Panel -->
                        <div id="perplexity-panel" class="hidden space-y-4">
                            <div class="space-y-3">
                                <input type="text" id="search-query" 
                                       placeholder="Search for information..." 
                                       class="w-full p-4 border border-gray-200 rounded-lg text-sm focus:ring-2 focus:ring-orange-500 focus:border-transparent">
                                <button id="search-submit" class="w-full bg-orange-500 text-white py-3 rounded-lg text-sm font-medium hover:bg-orange-600 transition-colors">
                                    <i class="fas fa-search mr-2"></i>Search
                                </button>
                            </div>

                            <!-- Results -->
                            <div id="search-results" class="hidden mt-4 p-4 bg-orange-50 rounded-lg">
                                <div class="flex items-center justify-between mb-3">
                                    <span class="font-medium text-orange-700">Search Results</span>
                                    <button id="search-close" class="text-gray-400 hover:text-gray-600">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                <div id="search-content" class="text-gray-700 text-sm space-y-3 mb-3"></div>
                                <button id="search-insert" class="w-full bg-orange-500 text-white py-2 rounded-lg text-sm hover:bg-orange-600">
                                    Insert Text
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Save Modal -->
    <div id="save-modal" class="hidden fixed inset-0 modal-backdrop z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-2xl p-8 w-full max-w-md">
            <h3 class="text-2xl font-semibold mb-6 text-gray-800">Save Document</h3>
            <input type="text" id="save-title" placeholder="Document name..." 
                   class="w-full p-4 border border-gray-200 rounded-lg mb-6 focus:ring-2 focus:ring-blue-500 focus:border-transparent">
            <div class="flex justify-end space-x-3">
                <button id="save-cancel" class="px-6 py-3 text-gray-600 hover:text-gray-800 font-medium">Cancel</button>
                <button id="save-confirm" class="px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 font-medium">Save</button>
            </div>
        </div>
    </div>

    <!-- Share Modal -->
    <div id="share-modal" class="hidden fixed inset-0 modal-backdrop z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-2xl p-8 w-full max-w-md">
            <h3 class="text-2xl font-semibold mb-6 text-gray-800">Share Document</h3>
            <div class="space-y-4 mb-6">
                <label class="flex items-center">
                    <input type="radio" name="share-type" value="view" class="mr-3" checked>
                    <span>Read only</span>
                </label>
                <label class="flex items-center">
                    <input type="radio" name="share-type" value="edit" class="mr-3">
                    <span>Allow editing</span>
                </label>
                <input type="text" id="share-link" readonly 
                       class="w-full p-4 border border-gray-200 rounded-lg bg-gray-50" 
                       placeholder="Link will be generated here...">
            </div>
            <div class="flex justify-end space-x-3">
                <button id="share-cancel" class="px-6 py-3 text-gray-600 hover:text-gray-800 font-medium">Cancel</button>
                <button id="share-generate" class="px-6 py-3 bg-green-500 text-white rounded-lg hover:bg-green-600 font-medium">Generate Link</button>
                <button id="share-copy" class="hidden px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 font-medium">Copy</button>
            </div>
        </div>
    </div>

<script>
// Variables globales
let currentDocument = {{ isset($document_id) ? $document_id : 'null' }};
let selectedText = '';
let editor = document.getElementById('editor');
let autoSaveInterval;

// Charger le document si ID fourni
if (currentDocument) {
    loadDocument(currentDocument);
}

// Fonction de chargement de document
async function loadDocument(id) {
    try {
        const response = await fetch(`/api/documents/${id}`);
        if (response.ok) {
            const document = await response.json();
            document.getElementById('document-title-input').value = document.title;
            editor.innerHTML = document.content;
        }
    } catch (error) {
        console.error('Erreur chargement document:', error);
    }
}

// Auto-save
function startAutoSave() {
    autoSaveInterval = setInterval(async () => {
        const title = document.getElementById('document-title-input').value;
        const content = editor.innerHTML;
        
        if (title.trim() !== '' || content.trim() !== '') {
            await saveDocument(false);
        }
    }, 30000);
}

async function saveDocument(showNotif = true) {
    const title = document.getElementById('document-title-input').value || 'Untitled Document';
    const content = editor.innerHTML;

    try {
        const response = await fetch('/api/documents/save', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                title: title,
                content: content,
                document_id: currentDocument
            })
        });

        const result = await response.json();
        
        if (result.success) {
            currentDocument = result.document_id;
            if (showNotif) {
                showNotification('Document saved!', 'success');
            }
        }
    } catch (error) {
        console.error('Erreur sauvegarde:', error);
        if (showNotif) {
            showNotification('Save error', 'error');
        }
    }
}

// Gestion des onglets IA
document.getElementById('tab-claude').addEventListener('click', () => {
    document.getElementById('tab-claude').classList.add('bg-blue-500', 'text-white');
    document.getElementById('tab-claude').classList.remove('bg-gray-100', 'text-gray-600');
    document.getElementById('tab-perplexity').classList.remove('bg-blue-500', 'text-white');
    document.getElementById('tab-perplexity').classList.add('bg-gray-100', 'text-gray-600');
    document.getElementById('claude-panel').classList.remove('hidden');
    document.getElementById('perplexity-panel').classList.add('hidden');
});

document.getElementById('tab-perplexity').addEventListener('click', () => {
    document.getElementById('tab-perplexity').classList.add('bg-blue-500', 'text-white');
    document.getElementById('tab-perplexity').classList.remove('bg-gray-100', 'text-gray-600');
    document.getElementById('tab-claude').classList.remove('bg-blue-500', 'text-white');
    document.getElementById('tab-claude').classList.add('bg-gray-100', 'text-gray-600');
    document.getElementById('perplexity-panel').classList.remove('hidden');
    document.getElementById('claude-panel').classList.add('hidden');
});

// Gestion sélection de texte
editor.addEventListener('mouseup', function() {
    selectedText = window.getSelection().toString().trim();
});

// Barre d'outils
document.querySelectorAll('.toolbar-btn[data-action]').forEach(button => {
    button.addEventListener('click', function() {
        const action = this.dataset.action;
        executeCommand(action);
        this.classList.toggle('active');
    });
});

function executeCommand(action) {
    switch(action) {
        case 'bold':
            document.execCommand('bold');
            break;
        case 'italic':
            document.execCommand('italic');
            break;
        case 'underline':
            document.execCommand('underline');
            break;
        case 'h1':
            document.execCommand('formatBlock', false, 'h1');
            break;
        case 'h2':
            document.execCommand('formatBlock', false, 'h2');
            break;
        case 'h3':
            document.execCommand('formatBlock', false, 'h3');
            break;
        case 'bullet-list':
            document.execCommand('insertUnorderedList');
            break;
        case 'number-list':
            document.execCommand('insertOrderedList');
            break;
    }
    editor.focus();
}

// Fonctions Claude
document.getElementById('claude-submit').addEventListener('click', () => {
    const prompt = document.getElementById('claude-prompt').value;
    if (prompt.trim()) {
        callClaudeAPI(prompt, 'custom');
    }
});

// Boutons Claude prédéfinis
document.querySelector('[data-claude-action="grammar"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'grammar');
    } else {
        showNotification('Select text to fix', 'warning');
    }
});

document.querySelector('[data-claude-action="expand"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'expand');
    } else {
        showNotification('Select text to expand', 'warning');
    }
});

document.querySelector('[data-claude-action="summarize"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'summarize');
    } else {
        showNotification('Select text to summarize', 'warning');
    }
});

document.querySelector('[data-claude-action="rephrase"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'rephrase');
    } else {
        showNotification('Select text to rephrase', 'warning');
    }
});

async function callClaudeAPI(prompt, action) {
    try {
        showLoading('claude');
        
        const response = await fetch('/api/claude', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                prompt: prompt,
                selected_text: selectedText,
                action: action
            })
        });

        const result = await response.json();
        
        if (result.success) {
            showClaudeResponse(result.content);
        } else {
            showNotification('Claude API error', 'error');
        }
    } catch (error) {
        console.error('Erreur Claude API:', error);
        showNotification('Connection error', 'error');
    } finally {
        hideLoading('claude');
    }
}

function showClaudeResponse(content) {
    document.getElementById('claude-content').textContent = content;
    document.getElementById('claude-response').classList.remove('hidden');
}

document.getElementById('claude-close').addEventListener('click', () => {
    document.getElementById('claude-response').classList.add('hidden');
});

document.getElementById('claude-apply').addEventListener('click', () => {
    const claudeContent = document.getElementById('claude-content').textContent;
    if (selectedText && claudeContent) {
        const selection = window.getSelection();
        if (selection.rangeCount > 0) {
            const range = selection.getRangeAt(0);
            range.deleteContents();
            range.insertNode(document.createTextNode(claudeContent));
        }
        document.getElementById('claude-response').classList.add('hidden');
        selectedText = '';
    }
});

// Fonctions Perplexity
document.getElementById('search-submit').addEventListener('click', () => {
    const query = document.getElementById('search-query').value;
    if (query.trim()) {
        callPerplexityAPI(query);
    }
});

async function callPerplexityAPI(query) {
    try {
        showLoading('search');
        
        const response = await fetch('/api/search', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                query: query
            })
        });

        const result = await response.json();
        
        if (result.success) {
            showSearchResults(result);
        } else {
            showNotification('Search API error', 'error');
        }
    } catch (error) {
        console.error('Erreur recherche API:', error);
        showNotification('Connection error', 'error');
    } finally {
        hideLoading('search');
    }
}

function showSearchResults(result) {
    const container = document.getElementById('search-content');
    let html = '';
    
    if (result.summary) {
        html += `<div class="bg-blue-50 p-3 rounded mb-3">
            <div class="font-medium text-blue-900 mb-1">Summary</div>
            <div class="text-blue-800 text-sm">${result.summary}</div>
        </div>`;
    }
    
    result.results.forEach(item => {
        html += `<div class="border-b pb-2 mb-2 last:border-b-0">
            <div class="font-medium text-sm">${item.title}</div>
            <div class="text-xs text-gray-600 mb-1">${item.source}</div>
            <div class="text-sm text-gray-700">${item.snippet}</div>
        </div>`;
    });
    
    container.innerHTML = html;
    document.getElementById('search-results').classList.remove('hidden');
}

document.getElementById('search-close').addEventListener('click', () => {
    document.getElementById('search-results').classList.add('hidden');
});

document.getElementById('search-insert').addEventListener('click', () => {
    const summary = document.querySelector('#search-content .bg-blue-50 .text-blue-800');
    if (summary) {
        insertAtCursor(summary.textContent);
        document.getElementById('search-results').classList.add('hidden');
    }
});

// Modals
document.getElementById('save-btn').addEventListener('click', () => {
    document.getElementById('save-title').value = document.getElementById('document-title-input').value;
    document.getElementById('save-modal').classList.remove('hidden');
});

document.getElementById('save-confirm').addEventListener('click', async () => {
    const title = document.getElementById('save-title').value;
    document.getElementById('document-title-input').value = title;
    await saveDocument(true);
    document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('save-cancel').addEventListener('click', () => {
    document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('share-btn').addEventListener('click', () => {
    if (!currentDocument) {
        showNotification('Save document first', 'warning');
        return;
    }
    document.getElementById('share-modal').classList.remove('hidden');
});

document.getElementById('share-generate').addEventListener('click', async () => {
    const permission = document.querySelector('input[name="share-type"]:checked').value;
    
    try {
        const response = await fetch(`/api/documents/${currentDocument}/share`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                permission: permission
            })
        });

        const result = await response.json();
        
        if (result.success) {
            document.getElementById('share-link').value = result.share_url;
            document.getElementById('share-copy').classList.remove('hidden');
        }
    } catch (error) {
        console.error('Erreur partage:', error);
        showNotification('Share error', 'error');
    }
});

document.getElementById('share-copy').addEventListener('click', () => {
    const shareLink = document.getElementById('share-link');
    shareLink.select();
    document.execCommand('copy');
    showNotification('Link copied!', 'success');
});

document.getElementById('share-cancel').addEventListener('click', () => {
    document.getElementById('share-modal').classList.add('hidden');
    document.getElementById('share-link').value = '';
    document.getElementById('share-copy').classList.add('hidden');
});

// Fonctions utilitaires
function insertAtCursor(text) {
    const selection = window.getSelection();
    if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        const textNode = document.createTextNode(text);
        range.insertNode(textNode);
        range.setStartAfter(textNode);
        range.setEndAfter(textNode);
        selection.removeAllRanges();
        selection.addRange(range);
    } else {
        editor.appendChild(document.createTextNode(text));
    }
    editor.focus();
}

function showLoading(type) {
    const button = type === 'claude' ? 
        document.getElementById('claude-submit') : 
        document.getElementById('search-submit');
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Loading...';
}

function hideLoading(type) {
    const button = type === 'claude' ? 
        document.getElementById('claude-submit') : 
        document.getElementById('search-submit');
    button.disabled = false;
    button.innerHTML = type === 'claude' ? 
        'Send to Claude' : 
        '<i class="fas fa-search mr-2"></i>Search';
}

function showNotification(message, type = 'info') {
    const colors = {
        success: 'bg-green-500',
        error: 'bg-red-500',
        warning: 'bg-yellow-500',
        info: 'bg-blue-500'
    };
    
    const notification = document.createElement('div');
    notification.className = `fixed top-4 left-4 ${colors[type]} text-white px-6 py-3 rounded-lg shadow-lg z-50 transform transition-all duration-300`;
    notification.textContent = message;
    notification.style.transform = 'translateY(-100px)';
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.transform = 'translateY(0)';
    }, 100);
    
    setTimeout(() => {
        notification.style.transform = 'translateY(-100px)';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Démarrer l'auto-save
startAutoSave();

// Cleanup à la fermeture
window.addEventListener('beforeunload', () => {
    if (autoSaveInterval) {
        clearInterval(autoSaveInterval);
    }
});
</script>
</body>
</html>