<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$error = null;
$success = null;

if (isset($_GET['accept']) && is_numeric($_GET['accept'])) {
    $mission_id = $_GET['accept'];
    
    $mission = db_get_one("SELECT * FROM missions WHERE id = ?", [$mission_id]);
    
    if (!$mission) {
        $error = "Mission non trouvée.";
    } elseif ($mission['required_level'] > $user['level']) {
        $error = "Niveau requis: " . $mission['required_level'] . ". Tu n'as pas le niveau suffisant.";
    } else {
        $existing = db_get_one(
            "SELECT * FROM user_missions WHERE user_id = ? AND mission_id = ? AND status != 'failed'",
            [$_SESSION['user_id'], $mission_id]
        );
        
        if ($existing) {
            $error = "Tu as déjà accepté cette mission.";
        } else {
            $data = [
                'user_id' => $_SESSION['user_id'],
                'mission_id' => $mission_id,
                'status' => 'active',
                'progress' => 0,
                'started_at' => date('Y-m-d H:i:s')
            ];
            
            db_insert('user_missions', $data);
            $success = "Mission acceptée!";
        }
    }
} elseif (isset($_GET['complete']) && is_numeric($_GET['complete'])) {
    $user_mission_id = $_GET['complete'];
    
    $user_mission = db_get_one(
        "SELECT um.*, m.title, m.reward_exp, m.reward_currency, m.currency_id
         FROM user_missions um
         JOIN missions m ON um.mission_id = m.id
         WHERE um.id = ? AND um.user_id = ? AND um.status = 'active'",
        [$user_mission_id, $_SESSION['user_id']]
    );
    
    if (!$user_mission) {
        $error = "Mission non trouvée ou déjà terminée.";
    } else {
        $data = [
            'status' => 'completed'
        ];
        
        db_update('user_missions', $data, 'id = ?', [$user_mission_id]);
        
        // Ajouter l'expérience
        add_experience($_SESSION['user_id'], $user_mission['reward_exp']);
        
        // Ajouter la récompense en devise
        $transaction_data = [
            'sender_id' => null,
            'receiver_id' => $_SESSION['user_id'],
            'amount' => $user_mission['reward_currency'],
            'currency_id' => $user_mission['currency_id'],
            'type' => 'mission_reward',
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        db_insert('economy_transactions', $transaction_data);
        
        $success = "Mission complétée! Récompenses: " . $user_mission['reward_exp'] . " XP et " . 
                   number_format($user_mission['reward_currency'], 2) . " crédits.";
    }
} elseif (isset($_GET['abandon']) && is_numeric($_GET['abandon'])) {
    $user_mission_id = $_GET['abandon'];
    
    $user_mission = db_get_one(
        "SELECT um.*, m.title
         FROM user_missions um
         JOIN missions m ON um.mission_id = m.id
         WHERE um.id = ? AND um.user_id = ? AND um.status = 'active'",
        [$user_mission_id, $_SESSION['user_id']]
    );
    
    if (!$user_mission) {
        $error = "Mission non trouvée ou déjà terminée.";
    } else {
        $data = [
            'status' => 'failed'
        ];
        
        db_update('user_missions', $data, 'id = ?', [$user_mission_id]);
        $success = "Mission abandonnée.";
    }
}

$available_missions = get_available_missions($_SESSION['user_id']);
$active_missions = db_get_all(
    "SELECT um.*, m.title, m.description, m.difficulty, m.reward_exp, m.reward_currency, m.required_level
     FROM user_missions um
     JOIN missions m ON um.mission_id = m.id
     WHERE um.user_id = ? AND um.status = 'active'
     ORDER BY um.started_at DESC",
    [$_SESSION['user_id']]
);

$completed_missions = db_get_all(
    "SELECT um.*, m.title, m.difficulty, m.reward_exp, m.reward_currency
     FROM user_missions um
     JOIN missions m ON um.mission_id = m.id
     WHERE um.user_id = ? AND um.status = 'completed'
     ORDER BY um.started_at DESC
     LIMIT 10",
    [$_SESSION['user_id']]
);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Missions | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
        .mission-card {
            transition: transform 0.2s, box-shadow 0.2s;
        }
        .mission-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 15px rgba(0, 0, 0, 0.1);
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300 font-semibold">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                    </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Missions</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Missions actives</h3>
                    
                    <?php if (empty($active_missions)): ?>
                        <p class="text-gray-500 italic">Tu n'as aucune mission active en ce moment.</p>
                    <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($active_missions as $mission): ?>
                                <?php
                                    $difficulty_colors = [
                                        'easy' => 'border-green-500 bg-green-50',
                                        'medium' => 'border-yellow-500 bg-yellow-50',
                                        'hard' => 'border-orange-500 bg-orange-50',
                                        'expert' => 'border-red-500 bg-red-50'
                                    ];
                                    $difficulty_text = [
                                        'easy' => 'Facile',
                                        'medium' => 'Moyenne',
                                        'hard' => 'Difficile',
                                        'expert' => 'Expert'
                                    ];
                                    $border_color = $difficulty_colors[$mission['difficulty']] ?? 'border-gray-300 bg-gray-50';
                                    $difficulty = $difficulty_text[$mission['difficulty']] ?? $mission['difficulty'];
                                ?>
                                <div class="mission-card border-l-4 rounded-lg p-4 <?php echo $border_color; ?>">
                                    <div class="flex justify-between items-start">
                                        <h4 class="font-medium"><?php echo htmlspecialchars($mission['title']); ?></h4>
                                        <span class="inline-block px-2 py-1 text-xs rounded bg-white bg-opacity-70">
                                            <?php echo $difficulty; ?>
                                        </span>
                                    </div>
                                    <p class="text-sm mt-2"><?php echo htmlspecialchars($mission['description']); ?></p>
                                    
                                    <div class="mt-3 text-sm flex items-center">
                                        <span class="font-medium mr-2">Progression:</span>
                                        <div class="flex-grow h-2 bg-white rounded-full overflow-hidden">
                                            <div class="h-full bg-blue-600" style="width: <?php echo $mission['progress']; ?>%"></div>
                                        </div>
                                        <span class="ml-2"><?php echo $mission['progress']; ?>%</span>
                                    </div>
                                    
                                    <div class="mt-3 text-sm">
                                        <span class="font-medium">Récompenses:</span> 
                                        <?php echo $mission['reward_exp']; ?> XP, 
                                        <?php echo number_format($mission['reward_currency'], 2); ?> crédits
                                    </div>
                                    
                                    <div class="mt-3 flex space-x-2">
                                        <?php if ($mission['progress'] >= 100): ?>
                                            <a href="missions.php?complete=<?php echo $mission['id']; ?>" class="inline-block bg-green-600 hover:bg-green-700 text-white text-sm py-1 px-3 rounded transition-colors">
                                                Compléter
                                            </a>
                                        <?php else: ?>
                                            <button type="button" class="inline-block bg-blue-600 hover:bg-blue-700 text-white text-sm py-1 px-3 rounded transition-colors" onclick="updateProgress(<?php echo $mission['id']; ?>)">
                                                Mettre à jour
                                            </button>
                                        <?php endif; ?>
                                        
                                        <a href="missions.php?abandon=<?php echo $mission['id']; ?>" class="inline-block bg-red-600 hover:bg-red-700 text-white text-sm py-1 px-3 rounded transition-colors" onclick="return confirm('Es-tu sûr de vouloir abandonner cette mission?')">
                                            Abandonner
                                        </a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Missions disponibles</h3>
                    
                    <?php if (empty($available_missions)): ?>
                        <p class="text-gray-500 italic">Aucune mission disponible pour le moment. Reviens plus tard!</p>
                    <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($available_missions as $mission): ?>
                                <?php
                                    $difficulty_colors = [
                                        'easy' => 'border-green-500',
                                        'medium' => 'border-yellow-500',
                                        'hard' => 'border-orange-500',
                                        'expert' => 'border-red-500'
                                    ];
                                    $difficulty_text = [
                                        'easy' => 'Facile',
                                        'medium' => 'Moyenne',
                                        'hard' => 'Difficile',
                                        'expert' => 'Expert'
                                    ];
                                    $border_color = $difficulty_colors[$mission['difficulty']] ?? 'border-gray-300';
                                    $difficulty = $difficulty_text[$mission['difficulty']] ?? $mission['difficulty'];
                                ?>
                                <div class="mission-card border-l-4 rounded-lg p-4 <?php echo $border_color; ?>">
                                    <div class="flex justify-between items-start">
                                        <h4 class="font-medium"><?php echo htmlspecialchars($mission['title']); ?></h4>
                                        <span class="inline-block px-2 py-1 text-xs rounded bg-gray-100">
                                            <?php echo $difficulty; ?>
                                        </span>
                                    </div>
                                    <p class="text-sm mt-2"><?php echo htmlspecialchars($mission['description']); ?></p>
                                    
                                    <div class="mt-3 text-sm">
                                        <span class="font-medium">Niveau requis:</span> 
                                        <?php echo $mission['required_level']; ?>
                                    </div>
                                    
                                    <div class="mt-1 text-sm">
                                        <span class="font-medium">Récompenses:</span> 
                                        <?php echo $mission['reward_exp']; ?> XP, 
                                        <?php echo number_format($mission['reward_currency'], 2); ?> crédits
                                    </div>
                                    
                                    <div class="mt-3">
                                        <a href="missions.php?accept=<?php echo $mission['id']; ?>" class="inline-block bg-blue-600 hover:bg-blue-700 text-white text-sm py-1 px-3 rounded transition-colors">
                                            Accepter la mission
                                        </a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div>
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Statut de l'agent</h3>
                    
                    <div class="space-y-3">
                        <div>
                            <p class="font-medium">Nom</p>
                            <p><?php echo htmlspecialchars($user['username']); ?></p>
                        </div>
                        <div>
                            <p class="font-medium">Niveau</p>
                            <p><?php echo $user['level']; ?> (<?php echo get_user_level_info($user['level'])['title']; ?>)</p>
                        </div>
                        <div>
                            <p class="font-medium">Expérience</p>
                            <p><?php echo $user['experience']; ?> / <?php echo get_user_level_info($user['level'])['max_exp']; ?></p>
                        </div>
                        <div>
                            <p class="font-medium">Missions actives</p>
                            <p><?php echo count($active_missions); ?></p>
                        </div>
                        <div>
                            <p class="font-medium">Missions complétées</p>
                            <p><?php echo count($completed_missions); ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Historique des missions</h3>
                    
                    <?php if (empty($completed_missions)): ?>
                        <p class="text-gray-500 italic">Tu n'as pas encore complété de mission.</p>
                    <?php else: ?>
                        <div class="space-y-2">
                            <?php foreach ($completed_missions as $mission): ?>
                                <div class="border-l-4 border-green-500 rounded-lg p-3 bg-green-50">
                                    <p class="font-medium"><?php echo htmlspecialchars($mission['title']); ?></p>
                                    <div class="flex justify-between text-sm">
                                        <span>Récompense: <?php echo $mission['reward_exp']; ?> XP</span>
                                        <span><?php echo format_datetime($mission['started_at']); ?></span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <script>
        function updateProgress(missionId) {
            const progressIncrement = Math.floor(Math.random() * 30) + 10; // 10-40% de progression
            
            fetch('update_mission_progress.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `mission_id=${missionId}&progress_increment=${progressIncrement}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Recharger la page pour afficher la progression mise à jour
                    location.reload();
                } else {
                    alert('Erreur: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                alert('Une erreur s\'est produite lors de la mise à jour de la progression.');
            });
        }
    </script>
</body>
</html>