const fs = require('fs');
const path = require('path');
const readline = require('readline');

const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout
});

function findAllFiles(directory) {
    let results = [];
    
    const items = fs.readdirSync(directory);
    
    for (const item of items) {
        const fullPath = path.join(directory, item);
        const stat = fs.statSync(fullPath);
        
        if (stat.isDirectory()) {
            results = results.concat(findAllFiles(fullPath));
        } else {
            if (path.extname(fullPath) === '.php') {
                results.push(fullPath);
            }
        }
    }
    
    return results;
}

function replaceFooterTag(filePath, newFooterContent) {
    try {
        let content = fs.readFileSync(filePath, 'utf8');
        
        // Expression régulière plus flexible pour capturer les balises footer
        // avec espaces, attributs, sauts de ligne, etc.
        const footerRegex = /<footer[^>]*>([\s\S]*?)<\/footer>/gi;
        
        // Teste si l'expression régulière trouve des correspondances
        const hasFooter = footerRegex.test(content);
        
        // Réinitialiser la regex car test() avance l'index
        const footerRegexForReplace = /<footer[^>]*>([\s\S]*?)<\/footer>/gi;
        
        if (hasFooter) {
            // Affiche la correspondance pour le débogage
            const matches = content.match(/<footer[^>]*>([\s\S]*?)<\/footer>/gi) || [];
            console.log(`Fichier: ${filePath}`);
            console.log(`Nombre de balises footer trouvées: ${matches.length}`);
            
            const modifiedContent = content.replace(footerRegexForReplace, (match, p1, offset) => {
                // Extrait la partie <footer ...> pour préserver les attributs
                const openingTag = match.substring(0, match.indexOf('>') + 1);
                return `${openingTag}${newFooterContent}</footer>`;
            });
            
            return {
                path: filePath,
                hasFooter: true,
                originalContent: content,
                modifiedContent: modifiedContent
            };
        } else {
            return {
                path: filePath,
                hasFooter: false
            };
        }
    } catch (error) {
        console.log(`Erreur lors du traitement du fichier ${filePath}: ${error.message}`);
        return {
            path: filePath,
            hasFooter: false,
            error: error.message
        };
    }
}

function processFiles() {
    const currentDir = process.cwd();
    const files = findAllFiles(currentDir);
    
    console.log(`Trouvé ${files.length} fichiers PHP dans le répertoire et ses sous-répertoires.`);
    
    let newFooterContent;
    
    try {
        const footerFilePath = path.join(currentDir, 'footer-maj.txt');
        if (fs.existsSync(footerFilePath)) {
            newFooterContent = fs.readFileSync(footerFilePath, 'utf8');
            console.log('Contenu du nouveau footer chargé depuis footer-maj.txt');
            console.log('Longueur du contenu: ' + newFooterContent.length + ' caractères');
            processFilesWithFooter(files, newFooterContent);
        } else {
            console.log('Le fichier footer-maj.txt n\'existe pas dans le répertoire courant.');
            rl.question('Veux-tu entrer le contenu du footer manuellement ? (yes/no) : ', (answer) => {
                if (answer.toLowerCase() === 'yes') {
                    rl.question('Entre le nouveau contenu pour les balises footer : ', (content) => {
                        newFooterContent = content;
                        processFilesWithFooter(files, newFooterContent);
                    });
                } else {
                    console.log('Opération annulée.');
                    rl.close();
                }
            });
        }
    } catch (error) {
        console.log('Erreur lors de la lecture du fichier footer-maj.txt:', error.message);
        rl.close();
    }
}

function processFilesWithFooter(files, newFooterContent) {
    const filesWithFooter = [];
    
    for (const file of files) {
        const result = replaceFooterTag(file, newFooterContent);
        
        if (result.hasFooter) {
            filesWithFooter.push(result);
        }
    }
    
    console.log(`Trouvé ${filesWithFooter.length} fichiers PHP contenant des balises footer.`);
    
    if (filesWithFooter.length === 0) {
        console.log('Aucun fichier PHP avec des balises footer trouvé. Fin du programme.');
        rl.close();
        return;
    }
    
    processNextFile(filesWithFooter, 0);
}

function processNextFile(filesWithFooter, index) {
    if (index >= filesWithFooter.length) {
        console.log('Tous les fichiers ont été traités. Fin du programme.');
        rl.close();
        return;
    }
    
    const file = filesWithFooter[index];
    console.log(`\nFichier ${index + 1}/${filesWithFooter.length}: ${file.path}`);
    
    rl.question('Modifier ce fichier ? (yes/no) : ', (answer) => {
        if (answer.toLowerCase() === 'yes') {
            fs.writeFileSync(file.path, file.modifiedContent, 'utf8');
            console.log('Fichier modifié avec succès.');
        } else {
            console.log('Fichier non modifié.');
        }
        
        processNextFile(filesWithFooter, index + 1);
    });
}

processFiles();