<?php
require_once 'config.php';

if(isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

if(isset($_POST['signup'])) {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $adult_check = isset($_POST['adult']) ? 1 : 0;

    // Validations
    if(strlen($username) < 3) {
        $error = "Username must be at least 3 characters long";
    } elseif(strlen($password) < 6) {
        $error = "Password must be at least 6 characters long";
    } elseif(!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "Please enter a valid email address";
    } elseif(!$adult_check) {
        $error = "You must confirm that you are over 18 years old";
    } else {
        try {
            // Check if username exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
            $stmt->execute([$username]);
            if($stmt->fetch()) {
                $error = "Username already exists";
            } else {
                // Check if email exists
                $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
                $stmt->execute([$email]);
                if($stmt->fetch()) {
                    $error = "Email already exists";
                } else {
                    // Create user
                    $stmt = $pdo->prepare("INSERT INTO users (username, email, password) VALUES (?, ?, ?)");
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    $stmt->execute([$username, $email, $hashedPassword]);
                    
                    // Auto login after signup
                    $_SESSION['user_id'] = $pdo->lastInsertId();
                    $_SESSION['username'] = $username;
                    header('Location: index.php');
                    exit;
                }
            }
        } catch(PDOException $e) {
            $error = "An error occurred. Please try again later.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sign Up - ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Titillium+Web:wght@200;300;400;600;700;900&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Titillium Web', sans-serif;
        }
        .formore {
            font-weight: 900;
            background: linear-gradient(to right, #3b82f6, #1d4ed8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        .form-container {
            animation: fadeIn 0.5s ease-out;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white">
    <div class="min-h-screen flex flex-col items-center justify-center p-4">
        <a href="index.php" class="mb-8">
            <h1 class="text-5xl formore">ForMore</h1>
        </a>

        <div class="form-container w-full max-w-md bg-gray-800/50 backdrop-blur-sm rounded-2xl p-8">
            <?php if(isset($error)): ?>
                <div class="bg-red-500/80 backdrop-blur-sm text-white p-4 rounded-xl mb-6 text-center">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST" class="space-y-6">
                <div>
                    <input type="text" name="username" required minlength="3" 
                           class="w-full bg-gray-900/50 backdrop-blur-sm border-0 rounded-xl p-4 text-white focus:ring-2 focus:ring-blue-500"
                           placeholder="Choose your username"
                           value="<?= isset($_POST['username']) ? htmlspecialchars($_POST['username']) : '' ?>">
                </div>

                <div>
                    <input type="email" name="email" required 
                           class="w-full bg-gray-900/50 backdrop-blur-sm border-0 rounded-xl p-4 text-white focus:ring-2 focus:ring-blue-500"
                           placeholder="Enter your email"
                           value="<?= isset($_POST['email']) ? htmlspecialchars($_POST['email']) : '' ?>">
                </div>

                <div>
                    <input type="password" name="password" required minlength="6" 
                           class="w-full bg-gray-900/50 backdrop-blur-sm border-0 rounded-xl p-4 text-white focus:ring-2 focus:ring-blue-500"
                           placeholder="Choose a password">
                </div>

                <div class="flex items-start gap-3">
                    <input type="checkbox" name="adult" required id="adult" 
                           class="mt-1.5 w-4 h-4 rounded border-gray-600 bg-gray-900/50 text-blue-500 focus:ring-blue-500">
                    <label for="adult" class="text-sm text-gray-300">
                        I confirm that I am over 18 years old and understand that ForMore is a platform intended for adult content only
                    </label>
                </div>

                <button type="submit" name="signup" 
                        class="w-full bg-blue-600 hover:bg-blue-700 text-white font-bold py-4 rounded-xl transition-colors">
                    Create Account
                </button>

                <p class="text-center text-gray-400">
                    Already have an account? 
                    <a href="login.php" class="text-blue-400 hover:text-blue-300 font-semibold">Sign in</a>
                </p>
            </form>
        </div>
    </div>
</body>
</html>