<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$country = get_user_country($_SESSION['user_id']);

$error = null;
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'add_combination') {
        $combination = $_POST['combination'] ?? '';
        
        if (empty($combination) || strlen($combination) !== 3 || !ctype_digit($combination)) {
            $error = "La combinaison doit être un nombre à 3 chiffres.";
        } else {
            $exists = db_get_value(
                "SELECT COUNT(*) FROM system_combinations WHERE country_id = ? AND combination = ? AND active = 1",
                [$country['id'], $combination]
            );
            
            if ($exists > 0) {
                $error = "Cette combinaison est déjà enregistrée dans le système.";
            } else {
                add_system_combination($country['id'], $combination, $_SESSION['user_id']);
                add_experience($_SESSION['user_id'], 50);
                $success = "Combinaison ajoutée avec succès! +50 XP";
            }
        }
    } elseif (isset($_POST['action']) && $_POST['action'] === 'attack_system') {
        $target_country = $_POST['target_country'] ?? '';
        $combination = $_POST['attack_combination'] ?? '';
        
        if (empty($target_country)) {
            $error = "Tu dois sélectionner un pays cible.";
        } elseif (empty($combination) || strlen($combination) !== 3 || !ctype_digit($combination)) {
            $error = "La combinaison d'attaque doit être un nombre à 3 chiffres.";
        } elseif ($target_country == $country['id']) {
            $error = "Tu ne peux pas attaquer ton propre pays.";
        } else {
            $result = attack_country_system($_SESSION['user_id'], $target_country, $combination);
            
            if ($result['success']) {
                $success = $result['message'];
            } else {
                $error = $result['message'];
            }
        }
    }
}

$combinations = db_get_all(
    "SELECT sc.*, u.username 
     FROM system_combinations sc
     JOIN users u ON sc.added_by = u.id
     WHERE sc.country_id = ? AND sc.active = 1
     ORDER BY sc.created_at DESC",
    [$country['id']]
);

$enemy_countries = db_get_all(
    "SELECT id, name, system_security_level 
     FROM countries 
     WHERE id != ? 
     ORDER BY name",
    [$country['id']]
);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Système national | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .terminal {
            background-color: #0D1117;
            color: #E6EDF3;
            font-family: 'Courier New', monospace;
            border-radius: 8px;
            overflow: hidden;
        }
        .terminal-header {
            background-color: #161B22;
            padding: 8px 16px;
            display: flex;
            align-items: center;
        }
        .terminal-circle {
            height: 12px;
            width: 12px;
            border-radius: 50%;
            margin-right: 6px;
        }
        .terminal-title {
            margin-left: 8px;
            font-size: 0.85rem;
            color: #8B949E;
        }
        .terminal-body {
            padding: 16px;
            min-height: 300px;
        }
        .terminal-line {
            line-height: 1.5;
        }
        .terminal-prompt {
            color: #7EE787;
        }
        .terminal-input-line {
            display: flex;
        }
        .blink {
            animation: blink-animation 1s steps(2, start) infinite;
        }
        @keyframes blink-animation {
            to {
                visibility: hidden;
            }
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300 font-semibold">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Système national: <?php echo htmlspecialchars($country['name']); ?></h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card mb-6">
                    <div class="p-6">
                        <h3 class="text-xl font-semibold mb-4">État du système</h3>
                        <div class="space-y-4">
                            <div>
                                <p class="mb-2 font-medium">Niveau de sécurité</p>
                                <div class="w-full bg-gray-200 rounded-full h-4">
                                    <div class="bg-green-600 h-4 rounded-full" style="width: <?php echo ($country['system_security_level'] / 5) * 100; ?>%"></div>
                                </div>
                                <p class="mt-1 text-sm text-gray-600">Niveau <?php echo $country['system_security_level']; ?> sur 5</p>
                            </div>
                            
                            <div>
                                <p class="mb-2 font-medium">Combinaisons sécurisées</p>
                                <p class="text-2xl"><?php echo count($combinations); ?> <span class="text-sm text-gray-600">/ 1000</span></p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="terminal">
                    <div class="terminal-header">
                        <div class="terminal-circle bg-red-500"></div>
                        <div class="terminal-circle bg-yellow-500"></div>
                        <div class="terminal-circle bg-green-500"></div>
                        <div class="terminal-title">Terminal système - <?php echo htmlspecialchars($country['name']); ?></div>
                    </div>
                    <div class="terminal-body">
                        <div class="terminal-line">Bienvenue sur le terminal de sécurité national</div>
                        <div class="terminal-line">Version 3.7.2 - Cergies Security Protocol</div>
                        <div class="terminal-line">--------------------------------------------</div>
                        <div class="terminal-line">État du système: <?php echo $country['system_security_level'] >= 4 ? 'SÉCURISÉ' : ($country['system_security_level'] >= 2 ? 'VULNÉRABLE' : 'CRITIQUE'); ?></div>
                        <div class="terminal-line">Niveau de sécurité: <?php echo $country['system_security_level']; ?>/5</div>
                        <div class="terminal-line">Combinaisons enregistrées: <?php echo count($combinations); ?></div>
                        <div class="terminal-line">--------------------------------------------</div>
                        <div class="terminal-line">Entrez une nouvelle combinaison pour renforcer la sécurité:</div>
                        <div class="terminal-input-line mt-2">
                            <form method="POST" action="system.php" class="flex w-full">
                                <input type="hidden" name="action" value="add_combination">
                                <span class="terminal-prompt mr-2">$</span>
                                <input type="text" name="combination" maxlength="3" pattern="[0-9]{3}" class="bg-transparent focus:outline-none flex-grow" placeholder="000-999" required>
                                <button type="submit" class="bg-green-600 text-white px-4 py-1 rounded">Ajouter</button>
                            </form>
                        </div>
                        
                        <div class="terminal-line mt-4">Ou attaquer un système ennemi:</div>
                        <div class="terminal-input-line mt-2">
                            <form method="POST" action="system.php" class="flex w-full flex-col space-y-2">
                                <input type="hidden" name="action" value="attack_system">
                                <div class="flex">
                                    <span class="terminal-prompt mr-2">$</span>
                                    <select name="target_country" class="bg-gray-800 border border-gray-700 text-gray-200 rounded px-2 py-1" required>
                                        <option value="">Sélectionner un pays cible</option>
                                        <?php foreach ($enemy_countries as $enemy): ?>
                                            <option value="<?php echo $enemy['id']; ?>">
                                                <?php echo htmlspecialchars($enemy['name']); ?> (Niveau <?php echo $enemy['system_security_level']; ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="flex">
                                    <span class="terminal-prompt mr-2">$</span>
                                    <input type="text" name="attack_combination" maxlength="3" pattern="[0-9]{3}" class="bg-transparent border-b border-gray-700 focus:outline-none focus:border-green-500" placeholder="Combinaison d'attaque (000-999)" required>
                                    <button type="submit" class="bg-red-600 text-white px-4 py-1 rounded ml-2">Attaquer</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
            
            <div>
                <div class="card p-6">
                    <h3 class="text-xl font-semibold mb-4">Combinaisons sécurisées</h3>
                    
                    <?php if (empty($combinations)): ?>
                        <p class="text-gray-500 italic">Aucune combinaison n'a encore été enregistrée. Ajoute-en pour renforcer la sécurité du système!</p>
                    <?php else: ?>
                        <div class="overflow-y-auto max-h-96">
                            <table class="w-full">
                                <thead>
                                    <tr>
                                        <th class="text-left px-2 py-2 border-b">Code</th>
                                        <th class="text-left px-2 py-2 border-b">Ajouté par</th>
                                        <th class="text-left px-2 py-2 border-b">Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($combinations as $comb): ?>
                                        <tr class="hover:bg-gray-50">
                                            <td class="px-2 py-2 border-b font-mono"><?php echo $comb['combination']; ?></td>
                                            <td class="px-2 py-2 border-b"><?php echo htmlspecialchars($comb['username']); ?></td>
                                            <td class="px-2 py-2 border-b text-sm text-gray-600">
                                                <?php echo format_datetime($comb['created_at']); ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('input[pattern]').forEach(input => {
                input.addEventListener('input', function() {
                    this.value = this.value.replace(/[^0-9]/g, '');
                    if (this.value.length > this.maxLength) {
                        this.value = this.value.slice(0, this.maxLength);
                    }
                });
            });
        });
    </script>
</body>
</html>