<?php 
require_once 'includes/auth.php';

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $age_rating = (int)$_POST['age_rating'];
    $author = trim($_POST['author']);
    $director = trim($_POST['director']);
    $terms_accepted = isset($_POST['terms_accepted']);
    $notice_accepted = isset($_POST['notice_accepted']);
    
    if (empty($title) || empty($description) || empty($author) || empty($director)) {
        $error = 'Please fill in all required fields';
    } elseif (!$terms_accepted) {
        $error = 'You must accept the Terms of Service';
    } elseif (!$notice_accepted) {
        $error = 'You must accept the content review notice';
    } elseif (!isset($_FILES['video']) || $_FILES['video']['error'] !== UPLOAD_ERR_OK) {
        $error = 'Please select a valid video file';
    } elseif (!isset($_FILES['thumbnail']) || $_FILES['thumbnail']['error'] !== UPLOAD_ERR_OK) {
        $error = 'Please select a valid thumbnail image';
    } else {
        $video_file = $_FILES['video'];
        $thumbnail_file = $_FILES['thumbnail'];
        
        $max_video_size = 500 * 1024 * 1024;
        $max_thumbnail_size = 5 * 1024 * 1024;
        
        $allowed_video_types = ['video/mp4', 'video/avi', 'video/mov', 'video/wmv', 'video/webm'];
        $allowed_image_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        
        if ($video_file['size'] > $max_video_size) {
            $error = 'Video file is too large. Maximum size is 500MB';
        } elseif ($thumbnail_file['size'] > $max_thumbnail_size) {
            $error = 'Thumbnail image is too large. Maximum size is 5MB';
        } elseif (!in_array($video_file['type'], $allowed_video_types)) {
            $error = 'Invalid video format. Please use MP4, AVI, MOV, WMV, or WebM';
        } elseif (!in_array($thumbnail_file['type'], $allowed_image_types)) {
            $error = 'Invalid thumbnail format. Please use JPEG, PNG, GIF, or WebP';
        } else {
            $upload_dir = 'uploads/';
            $video_dir = $upload_dir . 'videos/';
            $thumbnail_dir = $upload_dir . 'thumbnails/';
            
            if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
            if (!is_dir($video_dir)) mkdir($video_dir, 0755, true);
            if (!is_dir($thumbnail_dir)) mkdir($thumbnail_dir, 0755, true);
            
            $video_ext = pathinfo($video_file['name'], PATHINFO_EXTENSION);
            $thumbnail_ext = pathinfo($thumbnail_file['name'], PATHINFO_EXTENSION);
            
            $video_filename = uniqid() . '_' . time() . '.' . $video_ext;
            $thumbnail_filename = uniqid() . '_' . time() . '.' . $thumbnail_ext;
            
            $video_path = $video_dir . $video_filename;
            $thumbnail_path = $thumbnail_dir . $thumbnail_filename;
            
            if (move_uploaded_file($video_file['tmp_name'], $video_path) && 
                move_uploaded_file($thumbnail_file['tmp_name'], $thumbnail_path)) {
                
                $stmt = $pdo->prepare("
                    INSERT INTO content (title, description, thumbnail, video_url, age_rating, author, director, status, uploaded_by) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?)
                ");
                
                if ($stmt->execute([$title, $description, $thumbnail_path, $video_path, $age_rating, $author, $director, $current_user['id']])) {
                    $success = 'Content uploaded successfully! It will be reviewed within 48 hours.';
                } else {
                    unlink($video_path);
                    unlink($thumbnail_path);
                    $error = 'Failed to save content information';
                }
            } else {
                $error = 'Failed to upload files';
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Content - Formore</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <link rel="icon" href="https://cdn.imators.com/logo.png" type="image/png">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            font-weight: 300;
            background: #fff;
            color: #333;
            overflow-x: hidden;
        }
        
        .header {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 100;
            padding: 1rem 2rem;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .logo {
            font-size: 1.5rem;
            font-weight: 500;
            color: #000;
            text-decoration: none;
        }
        
        .back-btn {
            background: none;
            color: #333;
            border: none;
            padding: 0.5rem 1rem;
            cursor: pointer;
            font-size: 0.9rem;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 300;
            border-radius: 6px;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .back-btn:hover {
            background: #f8f9fa;
            color: #000;
        }
        
        .upload-container {
            max-width: 700px;
            margin: 120px auto 4rem;
            padding: 2rem;
        }

        .upload-form {
            background: #fff;
            border: 1px solid rgba(0, 0, 0, 0.05);
            border-radius: 12px;
            padding: 2.5rem;
        }
        
        .form-title {
            font-size: 1.75rem;
            font-weight: 500;
            margin-bottom: 0.5rem;
            color: #000;
            text-align: center;
        }
        
        .form-subtitle {
            color: #666;
            text-align: center;
            margin-bottom: 2rem;
            font-weight: 300;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 400;
            color: #333;
        }
        
        .form-input,
        .form-select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            background: #fff;
            font-size: 0.9rem;
            font-family: 'Poppins', sans-serif;
            font-weight: 300;
            transition: all 0.3s ease;
        }
        
        .form-input:focus,
        .form-select:focus {
            outline: none;
            border-color: #000;
            box-shadow: 0 0 0 3px rgba(0, 0, 0, 0.05);
        }
        
        .form-textarea {
            min-height: 100px;
            resize: vertical;
        }
        
        .file-upload {
            position: relative;
            display: block;
            cursor: pointer;
            border: 2px dashed #e9ecef;
            border-radius: 8px;
            padding: 2rem;
            text-align: center;
            background: #f8f9fa;
            transition: all 0.3s ease;
        }
        
        .file-upload:hover {
            border-color: #000;
            background: #fff;
        }
        
        .file-upload input {
            position: absolute;
            opacity: 0;
            pointer-events: none;
        }
        
        .file-upload-icon {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }
        
        .file-upload-text {
            color: #333;
            margin-bottom: 0.25rem;
            font-weight: 400;
        }
        
        .file-upload-hint {
            font-size: 0.85rem;
            color: #999;
        }
        
        .checkbox-group {
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
            margin-bottom: 1rem;
        }
        
        .checkbox {
            margin-top: 0.25rem;
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .checkbox-label {
            color: #333;
            line-height: 1.5;
            font-size: 0.9rem;
            font-weight: 300;
        }

        .checkbox-label a {
            color: #000;
            text-decoration: underline;
            transition: all 0.3s ease;
        }

        .checkbox-label a:hover {
            color: #666;
        }
        
        .submit-btn {
            width: 100%;
            background: #000;
            color: white;
            border: none;
            border-radius: 8px;
            padding: 0.75rem;
            font-size: 0.9rem;
            font-weight: 400;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 1rem;
            font-family: 'Poppins', sans-serif;
        }
        
        .submit-btn:hover {
            background: #333;
            transform: translateY(-1px);
        }
        
        .submit-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            font-size: 0.9rem;
        }
        
        .alert-error {
            background: rgba(255, 59, 48, 0.1);
            border: 1px solid rgba(255, 59, 48, 0.2);
            color: #d70015;
        }
        
        .alert-success {
            background: rgba(52, 199, 89, 0.1);
            border: 1px solid rgba(52, 199, 89, 0.2);
            color: #1d8348;
        }
        
        .progress-bar {
            width: 100%;
            height: 4px;
            background: #e9ecef;
            border-radius: 2px;
            overflow: hidden;
            margin-top: 0.5rem;
            display: none;
        }
        
        .progress-fill {
            height: 100%;
            background: #000;
            width: 0%;
            transition: width 0.3s ease;
        }

        .file-upload.active {
            border-color: #000;
            background: #f8f9fa;
        }

        .file-upload.active .progress-fill {
            background: #34c759;
        }

        @media (max-width: 768px) {
            .header {
                padding: 1rem;
            }

            .logo {
                font-size: 1.25rem;
            }

            .back-btn {
                padding: 0.5rem 0.75rem;
                font-size: 0.85rem;
            }

            .upload-container {
                margin-top: 80px;
                padding: 1rem;
            }

            .upload-form {
                padding: 1.5rem;
            }

            .form-title {
                font-size: 1.5rem;
            }

            .form-subtitle {
                font-size: 0.9rem;
                margin-bottom: 1.5rem;
            }

            .checkbox-label {
                font-size: 0.85rem;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <a href="home.php" class="logo">formore</a>
            <a href="home.php" class="back-btn">← Back</a>
        </div>
    </header>

    <div class="upload-container">
        <div class="upload-form">
            <h1 class="form-title">Upload Your Content</h1>
            <p class="form-subtitle">Share your movies and series made in video games</p>
            
            <?php if($error): ?>
                <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <?php if($success): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
            <?php endif; ?>
            
            <form method="POST" enctype="multipart/form-data" id="uploadForm">
                <div class="form-group">
                    <label class="form-label">Title *</label>
                    <input type="text" name="title" class="form-input" placeholder="Enter the title of your content" required value="<?php echo isset($_POST['title']) ? htmlspecialchars($_POST['title']) : ''; ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Description *</label>
                    <textarea name="description" class="form-input form-textarea" placeholder="Describe your content, the story, the game used..." required><?php echo isset($_POST['description']) ? htmlspecialchars($_POST['description']) : ''; ?></textarea>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label class="form-label">Author *</label>
                        <input type="text" name="author" class="form-input" placeholder="Content creator name" required value="<?php echo isset($_POST['author']) ? htmlspecialchars($_POST['author']) : ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Director *</label>
                        <input type="text" name="director" class="form-input" placeholder="Director name" required value="<?php echo isset($_POST['director']) ? htmlspecialchars($_POST['director']) : ''; ?>">
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Age Rating *</label>
                    <select name="age_rating" class="form-select" required>
                        <option value="6" <?php echo (isset($_POST['age_rating']) && $_POST['age_rating'] == '6') ? 'selected' : ''; ?>>6+ (Kids)</option>
                        <option value="13" <?php echo (isset($_POST['age_rating']) && $_POST['age_rating'] == '13') ? 'selected' : ''; ?>>13+ (Teen)</option>
                        <option value="16" <?php echo (isset($_POST['age_rating']) && $_POST['age_rating'] == '16') ? 'selected' : ''; ?>>16+ (Mature)</option>
                        <option value="18" <?php echo (isset($_POST['age_rating']) && $_POST['age_rating'] == '18') ? 'selected' : ''; ?>>18+ (Adult)</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Video File * (Max 500MB)</label>
                    <label class="file-upload">
                        <input type="file" name="video" accept="video/*" required>
                        <div class="file-upload-icon">🎬</div>
                        <div class="file-upload-text">Click to select your video</div>
                        <div class="file-upload-hint">MP4, AVI, MOV, WMV, WebM</div>
                    </label>
                    <div class="progress-bar" id="videoProgress">
                        <div class="progress-fill"></div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Thumbnail Image * (Max 5MB)</label>
                    <label class="file-upload">
                        <input type="file" name="thumbnail" accept="image/*" required>
                        <div class="file-upload-icon">🖼️</div>
                        <div class="file-upload-text">Click to select thumbnail</div>
                        <div class="file-upload-hint">JPEG, PNG, GIF, WebP</div>
                    </label>
                    <div class="progress-bar" id="thumbnailProgress">
                        <div class="progress-fill"></div>
                    </div>
                </div>
                
                <div class="checkbox-group">
                    <input type="checkbox" name="terms_accepted" class="checkbox" id="terms" required>
                    <label for="terms" class="checkbox-label">
                        I accept the <a href="https://imators.com/terms-of-use">Terms of Service</a>
                    </label>
                </div>
                
                <div class="checkbox-group">
                    <input type="checkbox" name="notice_accepted" class="checkbox" id="notice" required>
                    <label for="notice" class="checkbox-label">
                        I accept that the content I publish will be reviewed, at the level of information such as title, age, description, if these infringe the rules imposed by Formore.
                    </label>
                </div>
                
                <button type="submit" class="submit-btn" id="submitBtn">
                    Upload Content
                </button>
            </form>
        </div>
    </div>

    <script>
        document.querySelectorAll('input[type="file"]').forEach(input => {
            input.addEventListener('change', function() {
                const label = this.closest('.file-upload');
                const textDiv = label.querySelector('.file-upload-text');
                const progressBar = label.parentNode.querySelector('.progress-bar');
                
                if (this.files.length > 0) {
                    const file = this.files[0];
                    const maxSize = this.name === 'video' ? 500 * 1024 * 1024 : 5 * 1024 * 1024;
                    
                    if (file.size > maxSize) {
                        alert(`File is too large. Maximum size is ${this.name === 'video' ? '500MB' : '5MB'}`);
                        this.value = '';
                        return;
                    }
                    
                    textDiv.textContent = file.name;
                    label.classList.add('active');
                    
                    progressBar.style.display = 'block';
                    const progressFill = progressBar.querySelector('.progress-fill');
                    let progress = 0;
                    const interval = setInterval(() => {
                        progress += Math.random() * 20;
                        if (progress >= 100) {
                            progress = 100;
                            clearInterval(interval);
                        }
                        progressFill.style.width = progress + '%';
                    }, 100);
                } else {
                    textDiv.textContent = this.name === 'video' ? 'Click to select your video' : 'Click to select thumbnail';
                    label.classList.remove('active');
                    progressBar.style.display = 'none';
                }
            });
        });
        
        document.getElementById('uploadForm').addEventListener('submit', function() {
            const submitBtn = document.getElementById('submitBtn');
            submitBtn.disabled = true;
            submitBtn.textContent = 'Uploading...';
        });
    </script>
</body>
</html>