<?php
error_reporting(E_ERROR | E_PARSE);
ini_set('display_errors', 0);

define('JSON_FILE', __DIR__ . 'articles.json');

$articles = [];
$categories = [];
$errorMessage = null;

try {
    if (file_exists(JSON_FILE)) {
        $jsonContent = file_get_contents(JSON_FILE);
        if ($jsonContent === false) {
            throw new Exception("Unable to read JSON file");
        }
        
        $articlesData = json_decode($jsonContent, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("JSON format error");
        }
        
        $articles = $articlesData['articles'] ?? [];
        $categories = !empty($articles) ? array_unique(array_column($articles, 'category')) : [];
    } else {
        $articles = [
            [
                "id" => 1,
                "category" => "Getting Started",
                "title" => "Welcome to Support Center",
                "description" => "Learn how to use our support center and find the help you need.",
                "author" => "Support Team",
                "updated_at" => date('Y-m-d')
            ]
        ];
        $categories = ["Getting Started"];
    }
} catch (Exception $e) {
    $errorMessage = "An error occurred while loading articles.";
    $articles = [];
    $categories = [];
}

function filterArticles($articles, $search = '', $selectedCategory = '') {
    if (empty($articles)) return [];
    
    return array_filter($articles, function($article) use ($search, $selectedCategory) {
        $matchesSearch = empty($search) || 
            stripos($article['title'] ?? '', $search) !== false || 
            stripos($article['description'] ?? '', $search) !== false;
            
        $matchesCategory = empty($selectedCategory) || 
            ($article['category'] ?? '') === $selectedCategory;
            
        return $matchesSearch && $matchesCategory;
    });
}

$searchQuery = htmlspecialchars($_GET['search'] ?? '', ENT_QUOTES, 'UTF-8');
$selectedCategory = htmlspecialchars($_GET['category'] ?? '', ENT_QUOTES, 'UTF-8');
$filteredArticles = filterArticles($articles, $searchQuery, $selectedCategory);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Support Center - Help Articles</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .article-card:hover .article-arrow { transform: translateX(5px); }
    </style>
</head>
<body class="bg-gray-900 text-white min-h-screen">
    <?php if ($errorMessage): ?>
    <div class="fixed top-4 right-4 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg">
        <?php echo $errorMessage; ?>
    </div>
    <?php endif; ?>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <header class="text-center mb-12">
            <a href="/" class="inline-block mb-8">
                <img src="https://cdn.imators.com/logo.png" alt="Imators Logo" class="h-12 mx-auto">
            </a>
            
            <h1 class="text-4xl font-bold mb-4">How can we help you?</h1>
            <div class="max-w-2xl mx-auto">
                <form action="" method="GET" class="relative">
                    <?php if($selectedCategory): ?>
                    <input type="hidden" name="category" value="<?php echo $selectedCategory; ?>">
                    <?php endif; ?>
                    <input type="text" name="search" value="<?php echo $searchQuery; ?>" placeholder="Search for articles..." 
                           class="w-full px-6 py-4 bg-gray-800 rounded-full text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-sky-400 pr-12"
                           autocomplete="off">
                    <button type="submit" class="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-white transition-colors">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                        </svg>
                    </button>
                </form>
            </div>
        </header>

        <div class="grid grid-cols-12 gap-8">
            <aside class="col-span-12 lg:col-span-3">
                <div class="bg-gray-800 rounded-2xl p-6 sticky top-6">
                    <h2 class="text-lg font-semibold mb-4">Categories</h2>
                    <nav class="space-y-2">
                        <a href="?" class="block px-4 py-2 rounded-lg <?php echo empty($selectedCategory) ? 'bg-sky-400 text-white' : 'text-gray-300 hover:bg-gray-700'; ?> transition-all">
                            All Articles
                        </a>
                        <?php foreach($categories as $category): ?>
                        <a href="?category=<?php echo urlencode($category); ?><?php echo $searchQuery ? '&search='.urlencode($searchQuery) : ''; ?>" 
                           class="block px-4 py-2 rounded-lg <?php echo $selectedCategory === $category ? 'bg-sky-400 text-white' : 'text-gray-300 hover:bg-gray-700'; ?> transition-all">
                            <?php echo htmlspecialchars($category); ?>
                        </a>
                        <?php endforeach; ?>
                    </nav>
                </div>
            </aside>

            <main class="col-span-12 lg:col-span-9">
                <?php if(empty($filteredArticles)): ?>
                <div class="bg-gray-800 rounded-2xl p-8 text-center">
                    <svg class="w-16 h-16 text-gray-600 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M12 12h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <h3 class="text-xl font-semibold mb-2">No articles found</h3>
                    <p class="text-gray-400">Try adjusting your search or filter to find what you're looking for.</p>
                </div>
                <?php else: ?>
                <div class="grid gap-6">
                    <?php foreach($filteredArticles as $article): ?>
                    <article class="article-card bg-gray-800 rounded-2xl p-6 hover:bg-gray-750 transition-all">
                        <a href="/article.php?id=<?php echo $article['id']; ?>" class="block">
                            <div class="flex items-start justify-between">
                                <div>
                                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-sky-400 bg-opacity-10 text-sky-400 mb-3">
                                        <?php echo htmlspecialchars($article['category'] ?? ''); ?>
                                    </span>
                                    <h3 class="text-xl font-semibold mb-2 group-hover:text-sky-400 transition-colors">
                                        <?php echo htmlspecialchars($article['title'] ?? ''); ?>
                                    </h3>
                                    <p class="text-gray-400 mb-4">
                                        <?php echo htmlspecialchars($article['description'] ?? ''); ?>
                                    </p>
                                    <div class="flex items-center text-sm text-gray-500">
                                        <span>Written by <?php echo htmlspecialchars($article['author'] ?? ''); ?></span>
                                        <span class="mx-2">•</span>
                                        <span>Last updated <?php echo date('M j, Y', strtotime($article['updated_at'] ?? '')); ?></span>
                                    </div>
                                </div>
                                <svg class="w-5 h-5 text-gray-400 flex-shrink-0 article-arrow transition-transform duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>
                                </svg>
                            </div>
                        </a>
                    </article>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const searchInput = document.querySelector('input[name="search"]');
        const form = searchInput.closest('form');
        let timeoutId;
        searchInput.addEventListener('input', function() {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => {
                form.submit();
            }, 300);
        });
    });
    </script>
</body>
</html>