<?php
session_start();
include 'db.php';

if (!isset($_SESSION['resident_id'])) {
    header('Location: login.php');
    exit;
}

if (isset($_GET['member_id'])) {
    $member_id = intval($_GET['member_id']);
    
    $stmt = $conn->prepare("SELECT id, first_name, last_name, email, address, member_, created_at FROM residents WHERE id = ?");
    $stmt->bind_param("i", $member_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $member = $result->fetch_assoc();
        echo json_encode([
            'success' => true,
            'member' => $member
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'error' => 'Membre introuvable'
        ]);
    }
    exit;
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Scanner Carte Membre - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/html5-qrcode"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {}
            }
        }
    </script>
    <style>
        body {
            font-family: 'Poppins', sans-serif;
        }
        
        #reader {
            max-width: 600px;
            margin: 0 auto;
            border-radius: 16px;
            overflow: hidden;
        }
        
        .member-card-fullscreen {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            background: linear-gradient(135deg, #059669 0%, #047857 100%);
            color: white;
            display: none;
            z-index: 9999;
            overflow-y: auto;
            overflow-x: hidden;
            -webkit-overflow-scrolling: touch;
        }

        .member-card-fullscreen.show {
            display: block;
        }

        .member-card-content {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            padding: 40px 20px;
        }
        
        .info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 20px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.15);
        }
        
        .info-row:last-child {
            border-bottom: none;
        }

        .badge {
            display: inline-block;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .badge-premium {
            background: #22c55e;
            color: white;
        }

        .badge-standard {
            background: rgba(255, 255, 255, 0.2);
            color: white;
        }

        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            background: rgba(0, 0, 0, 0.9);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        }

        .loading-overlay.show {
            display: flex;
        }

        .spinner {
            width: 60px;
            height: 60px;
            border: 4px solid rgba(255, 255, 255, 0.3);
            border-top-color: #22c55e;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .scanner-container {
            background: white;
            border-radius: 24px;
            padding: 24px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
        }

        .dark .scanner-container {
            background: #1f2937;
        }
    </style>
</head>
<body class="bg-white dark:bg-black min-h-screen transition-colors duration-300">
    <script>
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    </script>

    <div id="loadingOverlay" class="loading-overlay">
        <div class="text-center">
            <div class="spinner mx-auto mb-4"></div>
            <p class="text-white text-xl font-light">Vérification en cours...</p>
        </div>
    </div>

    <div id="scannerSection" class="max-w-5xl mx-auto px-4 py-8">
        <div class="text-center mb-12">
            <a href="/" class="inline-block mb-6 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 transition">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"/>
                </svg>
            </a>
            <h1 class="text-4xl md:text-5xl font-light text-black dark:text-white mb-3 tracking-tight">Scanner Carte Membre</h1>
            <p class="text-lg text-gray-600 dark:text-gray-400">Scanne le QR code pour vérifier le statut</p>
        </div>

        <div class="scanner-container max-w-2xl mx-auto">
            <div id="reader"></div>
        </div>
    </div>

    <div id="memberCardFullscreen" class="member-card-fullscreen">
        <div class="member-card-content">
            <div class="max-w-xl w-full">
                <div class="flex items-start justify-between mb-8">
                    <div>
                        <h2 id="fullMemberName" class="text-3xl md:text-4xl font-bold mb-3"></h2>
                        <p class="text-white/90 text-lg">Membre Vert Chasseur</p>
                    </div>
                    <span id="fullMemberBadge" class="badge"></span>
                </div>
                
                <div class="bg-white/10 backdrop-blur-sm rounded-3xl p-6 md:p-8 mb-8">
                    <div class="space-y-2">
                        <div class="info-row">
                            <span class="text-white/70 text-base md:text-lg">ID Membre</span>
                            <span id="fullMemberId" class="font-bold text-xl md:text-2xl"></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="text-white/70 text-base md:text-lg">Email</span>
                            <span id="fullMemberEmail" class="font-medium text-sm md:text-base text-right break-all"></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="text-white/70 text-base md:text-lg">Adresse</span>
                            <span id="fullMemberAddress" class="font-medium text-sm md:text-base text-right"></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="text-white/70 text-base md:text-lg">Membre depuis</span>
                            <span id="fullMemberSince" class="font-medium text-base md:text-lg"></span>
                        </div>
                    </div>
                </div>
                
                <div id="fullMemberStatus" class="backdrop-blur-sm rounded-3xl p-6 md:p-8 text-center mb-8"></div>
                
                <button onclick="nextScan()" class="w-full bg-white text-green-700 py-4 px-6 rounded-xl font-bold text-lg hover:bg-gray-100 transition-all transform hover:scale-105">
                    Suivant
                </button>
            </div>
        </div>
    </div>

    <script>
        let html5QrCode = null;

        function startScanner() {
            html5QrCode = new Html5Qrcode("reader");
            
            html5QrCode.start(
                { facingMode: "environment" },
                {
                    fps: 10,
                    qrbox: { width: 250, height: 250 }
                },
                onScanSuccess
            ).catch(err => {
                console.error('Erreur scanner:', err);
                alert('Impossible de démarrer le scanner. Vérifie les permissions de la caméra.');
            });
        }

        function onScanSuccess(decodedText) {
            console.log('QR scanné:', decodedText);
            
            if (html5QrCode) {
                html5QrCode.stop();
            }
            
            fetchMemberInfo(decodedText);
        }

        async function fetchMemberInfo(memberId) {
            document.getElementById('loadingOverlay').classList.add('show');
            
            try {
                const response = await fetch('scan-member.php?member_id=' + memberId);
                const data = await response.json();
                
                if (!data.success) {
                    throw new Error(data.error || 'Membre introuvable');
                }
                
                const member = data.member;
                
                document.getElementById('fullMemberName').textContent = member.first_name + ' ' + member.last_name;
                document.getElementById('fullMemberId').textContent = '#' + String(member.id).padStart(6, '0');
                document.getElementById('fullMemberEmail').textContent = member.email;
                document.getElementById('fullMemberAddress').textContent = member.address;
                document.getElementById('fullMemberSince').textContent = new Date(member.created_at).toLocaleDateString('fr-FR');
                
                const badge = document.getElementById('fullMemberBadge');
                const isPremium = member.member_ == 1;
                
                if (isPremium) {
                    badge.className = 'badge badge-premium';
                    badge.textContent = 'PREMIUM';
                    
                    document.getElementById('fullMemberStatus').className = 'bg-green-500/30 backdrop-blur-sm rounded-3xl p-6 md:p-8 text-center mb-8 border border-green-400/30';
                    document.getElementById('fullMemberStatus').innerHTML = `
                        <svg class="w-12 h-12 md:w-16 md:h-16 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <p class="font-bold text-xl md:text-2xl mb-2">Réduction applicable</p>
                        <p class="text-white/90 text-base md:text-lg">10% de réduction chez tous les partenaires</p>
                    `;
                } else {
                    badge.className = 'badge badge-standard';
                    badge.textContent = 'STANDARD';
                    
                    document.getElementById('fullMemberStatus').className = 'bg-yellow-500/30 backdrop-blur-sm rounded-3xl p-6 md:p-8 text-center mb-8 border border-yellow-400/30';
                    document.getElementById('fullMemberStatus').innerHTML = `
                        <svg class="w-12 h-12 md:w-16 md:h-16 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                        </svg>
                        <p class="font-bold text-xl md:text-2xl mb-2">Membre Standard</p>
                        <p class="text-white/90 text-base md:text-lg">Aucune réduction applicable</p>
                    `;
                }
                
                setTimeout(() => {
                    document.getElementById('loadingOverlay').classList.remove('show');
                    document.getElementById('memberCardFullscreen').classList.add('show');
                }, 800);
                
            } catch (error) {
                console.error('Erreur:', error);
                document.getElementById('loadingOverlay').classList.remove('show');
                alert('Erreur: ' + error.message);
                startScanner();
            }
        }

        function nextScan() {
            document.getElementById('memberCardFullscreen').classList.remove('show');
            startScanner();
        }

        document.addEventListener('DOMContentLoaded', function() {
            startScanner();
        });
    </script>
</body>
</html>