<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Imators Site Monitor</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'media'
        }
    </script>
</head>
<body class="bg-white dark:bg-gray-900 min-h-screen p-8 transition-colors">
    <div class="max-w-6xl mx-auto">
        <div class="flex items-center mb-8">
            <img src="https://imators.com/logo.png" alt="Imators" class="w-12 h-12 mr-4" onerror="this.style.display='none'">
            <h1 class="text-3xl font-bold text-gray-800 dark:text-white">Imators Site Monitor</h1>
        </div>
        
        <div class="bg-gray-50 dark:bg-gray-800 rounded-lg shadow-lg p-6">
            <div class="flex justify-between items-center mb-6">
                <div class="text-sm text-gray-600 dark:text-gray-400">
                    Last check: <span id="lastCheck" class="font-medium text-gray-800 dark:text-white">-</span>
                </div>
                <div class="text-sm text-gray-600 dark:text-gray-400">
                    Next check in: <span id="nextCheck" class="font-medium text-gray-800 dark:text-white">60s</span>
                </div>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="border-b border-gray-200 dark:border-gray-700">
                            <th class="text-left py-3 px-4 font-semibold text-gray-700 dark:text-gray-300">Site</th>
                            <th class="text-left py-3 px-4 font-semibold text-gray-700 dark:text-gray-300">Status</th>
                            <th class="text-left py-3 px-4 font-semibold text-gray-700 dark:text-gray-300">24h History</th>
                        </tr>
                    </thead>
                    <tbody id="sitesTable">
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script>
        const sites = [
            'imators.com',
            'imators.fr', 
            'imators.jp',
            'imators.cn',
            'imators.ch',
            'imators.be',
            'imators.ae',
            'imators.nl',
            'imators.co.uk',
            'imators.uk',
            'imators.ma',
            'imators.systems',
            'ohmypanel.com',
            'academ.my',
            'formore.tv'
        ];

        let siteData = {};
        let countdownInterval;
        let checkInterval;
        let useLocalStorage = false;

        async function apiCall(action, data = {}) {
            try {
                const response = await fetch(API_BASE, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({action, ...data})
                });
                return await response.json();
            } catch (error) {
                console.error('API Error:', error);
                return null;
            }
        }

        async function loadSiteData() {
            if (useLocalStorage) {
                const saved = localStorage.getItem('siteMonitorData');
                if (saved) {
                    siteData = JSON.parse(saved);
                } else {
                    sites.forEach(site => {
                        siteData[site] = {
                            status: 'unknown',
                            history: new Array(24).fill('unknown')
                        };
                    });
                }
            } else {
                const result = await apiCall('getSiteData');
                if (result && result.success) {
                    siteData = result.data;
                } else {
                    sites.forEach(site => {
                        siteData[site] = {
                            status: 'unknown',
                            history: new Array(24).fill('unknown')
                        };
                    });
                }
            }
        }

        async function saveSiteData() {
            if (useLocalStorage) {
                localStorage.setItem('siteMonitorData', JSON.stringify(siteData));
            } else {
                await apiCall('saveSiteData', {data: siteData});
            }
        }

        async function checkSite(site) {
            try {
                const proxyUrl = `https://api.allorigins.win/get?url=${encodeURIComponent(`https://${site}`)}`;
                const response = await fetch(proxyUrl);
                
                if (response.ok) {
                    const data = await response.json();
                    if (data.contents && data.contents.includes('<meta name="imators-verification" content="verified">')) {
                        return 'online';
                    }
                }
                return 'offline';
            } catch (error) {
                return 'offline';
            }
        }

        async function checkAllSites() {
            const currentHour = new Date().getHours();
            
            for (const site of sites) {
                const status = await checkSite(site);
                siteData[site].status = status;
                siteData[site].history[currentHour] = status;
            }
            
            await saveSiteData();
            updateDisplay();
            document.getElementById('lastCheck').textContent = new Date().toLocaleTimeString();
        }

        function updateDisplay() {
            const tbody = document.getElementById('sitesTable');
            tbody.innerHTML = '';

            sites.forEach(site => {
                const data = siteData[site];
                const row = document.createElement('tr');
                row.className = 'border-b border-gray-100 dark:border-gray-700 hover:bg-gray-100 dark:hover:bg-gray-700';

                const statusClass = data.status === 'online' ? 'bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200' : 
                                   data.status === 'offline' ? 'bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200' : 
                                   'bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-gray-200';

                const historyHTML = data.history.map((status, hour) => {
                    const colorClass = status === 'online' ? 'bg-green-500' : 
                                      status === 'offline' ? 'bg-red-500' : 
                                      'bg-gray-400 dark:bg-gray-600';
                    return `<div class="w-3 h-3 ${colorClass} rounded-sm mr-1" title="Hour ${hour}: ${status}"></div>`;
                }).join('');

                row.innerHTML = `
                    <td class="py-3 px-4 font-medium text-gray-900 dark:text-white">${site}</td>
                    <td class="py-3 px-4">
                        <span class="px-2 py-1 text-xs rounded-full ${statusClass}">${data.status}</span>
                    </td>
                    <td class="py-3 px-4">
                        <div class="flex items-center">${historyHTML}</div>
                    </td>
                `;

                tbody.appendChild(row);
            });
        }

        function getSecondsUntilNextMinute() {
            const now = new Date();
            return 60 - now.getSeconds();
        }

        function updateCountdown() {
            const secondsLeft = getSecondsUntilNextMinute();
            document.getElementById('nextCheck').textContent = `${secondsLeft}s`;
        }

        function startSyncedTimer() {
            updateCountdown();
            
            const msUntilNextMinute = getSecondsUntilNextMinute() * 1000;
            
            setTimeout(() => {
                checkAllSites();
                
                checkInterval = setInterval(() => {
                    checkAllSites();
                }, 60000);
                
            }, msUntilNextMinute);
            
            countdownInterval = setInterval(updateCountdown, 1000);
        }

        async function refreshData() {
            if (!useLocalStorage) {
                await loadSiteData();
                updateDisplay();
            }
        }

        setInterval(refreshData, 30000);

        loadSiteData().then(() => {
            updateDisplay();
            startSyncedTimer();
        });
    </script>
</body>
</html>