<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../db.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'get_apps':
            handleGetApps();
            break;
        case 'create_app':
            handleCreateApp();
            break;
        case 'update_app':
            handleUpdateApp();
            break;
        case 'delete_app':
            handleDeleteApp();
            break;
        case 'get_categories':
            handleGetCategories();
            break;
        case 'create_category':
            handleCreateCategory();
            break;
        case 'update_category':
            handleUpdateCategory();
            break;
        case 'delete_category':
            handleDeleteCategory();
            break;
        case 'get_developers':
            handleGetDevelopers();
            break;
        case 'create_developer':
            handleCreateDeveloper();
            break;
        case 'update_developer':
            handleUpdateDeveloper();
            break;
        case 'delete_developer':
            handleDeleteDeveloper();
            break;
        case 'get_stats':
            handleGetStats();
            break;
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid action']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function handleGetApps() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT 
            a.*,
            d.name as developer_name
        FROM apps a 
        LEFT JOIN developers d ON a.developer_id = d.id
        ORDER BY a.created_at DESC
    ");
    
    $stmt->execute();
    $apps = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode($apps);
}

function handleCreateApp() {
    global $pdo;
    
    $iconUrl = null;
    $downloadUrl = null;
    
    if (isset($_FILES['icon']) && $_FILES['icon']['error'] === UPLOAD_ERR_OK) {
        $iconUrl = uploadFile($_FILES['icon'], 'icons');
    }
    
    if (isset($_FILES['download_file']) && $_FILES['download_file']['error'] === UPLOAD_ERR_OK) {
        $downloadUrl = uploadFile($_FILES['download_file'], 'downloads');
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO apps (
            name, slug, description, long_description, price, price_type, 
            category, icon_url, download_url, purchase_url, version, 
            requirements, size_mb, status, featured, has_in_app_purchases, created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    
    $stmt->execute([
        $_POST['name'],
        $_POST['slug'],
        $_POST['description'],
        $_POST['long_description'] ?: null,
        floatval($_POST['price']) ?: 0,
        $_POST['price_type'],
        $_POST['category'],
        $iconUrl,
        $downloadUrl,
        $_POST['purchase_url'] ?: null,
        $_POST['version'] ?: '1.0.0',
        $_POST['requirements'] ?: null,
        floatval($_POST['size_mb']) ?: null,
        $_POST['status'] ?: 'active',
        isset($_POST['featured']) ? 1 : 0,
        isset($_POST['has_in_app_purchases']) ? 1 : 0
    ]);
    
    echo json_encode(['success' => true, 'id' => $pdo->lastInsertId()]);
}

function handleUpdateApp() {
    global $pdo;
    
    $id = intval($_POST['id']);
    
    $app = $pdo->prepare("SELECT * FROM apps WHERE id = ?");
    $app->execute([$id]);
    $existingApp = $app->fetch(PDO::FETCH_ASSOC);
    
    if (!$existingApp) {
        throw new Exception('App not found');
    }
    
    $iconUrl = $existingApp['icon_url'];
    $downloadUrl = $existingApp['download_url'];
    
    if (isset($_FILES['icon']) && $_FILES['icon']['error'] === UPLOAD_ERR_OK) {
        if ($iconUrl) {
            deleteFile($iconUrl);
        }
        $iconUrl = uploadFile($_FILES['icon'], 'icons');
    }
    
    if (isset($_FILES['download_file']) && $_FILES['download_file']['error'] === UPLOAD_ERR_OK) {
        if ($downloadUrl) {
            deleteFile($downloadUrl);
        }
        $downloadUrl = uploadFile($_FILES['download_file'], 'downloads');
    }
    
    $stmt = $pdo->prepare("
        UPDATE apps SET 
            name = ?, slug = ?, description = ?, long_description = ?, 
            price = ?, price_type = ?, category = ?, icon_url = ?, 
            download_url = ?, purchase_url = ?, version = ?, requirements = ?, 
            size_mb = ?, status = ?, featured = ?, has_in_app_purchases = ?, updated_at = NOW()
        WHERE id = ?
    ");
    
    $stmt->execute([
        $_POST['name'],
        $_POST['slug'],
        $_POST['description'],
        $_POST['long_description'] ?: null,
        floatval($_POST['price']) ?: 0,
        $_POST['price_type'],
        $_POST['category'],
        $iconUrl,
        $downloadUrl,
        $_POST['purchase_url'] ?: null,
        $_POST['version'] ?: '1.0.0',
        $_POST['requirements'] ?: null,
        floatval($_POST['size_mb']) ?: null,
        $_POST['status'] ?: 'active',
        isset($_POST['featured']) ? 1 : 0,
        isset($_POST['has_in_app_purchases']) ? 1 : 0,
        $id
    ]);
    
    echo json_encode(['success' => true]);
}

function handleDeleteApp() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $id = intval($input['id']);
    
    $app = $pdo->prepare("SELECT icon_url, download_url FROM apps WHERE id = ?");
    $app->execute([$id]);
    $existingApp = $app->fetch(PDO::FETCH_ASSOC);
    
    if ($existingApp) {
        if ($existingApp['icon_url']) {
            deleteFile($existingApp['icon_url']);
        }
        if ($existingApp['download_url']) {
            deleteFile($existingApp['download_url']);
        }
    }
    
    $stmt = $pdo->prepare("DELETE FROM apps WHERE id = ?");
    $stmt->execute([$id]);
    
    echo json_encode(['success' => true]);
}

function handleGetCategories() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT 
            c.*,
            COUNT(a.id) as apps_count
        FROM categories c 
        LEFT JOIN apps a ON c.slug = a.category
        GROUP BY c.id
        ORDER BY c.display_order, c.name
    ");
    $stmt->execute();
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode($categories);
}

function handleCreateCategory() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        INSERT INTO categories (name, slug, description, icon, display_order, status, created_at) 
        VALUES (?, ?, ?, ?, ?, 'active', NOW())
    ");
    
    $stmt->execute([
        $_POST['name'],
        $_POST['slug'],
        $_POST['description'] ?: null,
        $_POST['icon'] ?: 'fas fa-folder',
        intval($_POST['display_order']) ?: 0
    ]);
    
    echo json_encode(['success' => true, 'id' => $pdo->lastInsertId()]);
}

function handleUpdateCategory() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        UPDATE categories SET 
            name = ?, slug = ?, description = ?, icon = ?, display_order = ?
        WHERE id = ?
    ");
    
    $stmt->execute([
        $_POST['name'],
        $_POST['slug'],
        $_POST['description'] ?: null,
        $_POST['icon'] ?: 'fas fa-folder',
        intval($_POST['display_order']) ?: 0,
        intval($_POST['id'])
    ]);
    
    echo json_encode(['success' => true]);
}

function handleDeleteCategory() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $id = intval($input['id']);
    
    $stmt = $pdo->prepare("DELETE FROM categories WHERE id = ?");
    $stmt->execute([$id]);
    
    echo json_encode(['success' => true]);
}

function handleGetDevelopers() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT 
            d.*,
            COUNT(a.id) as apps_count
        FROM developers d 
        LEFT JOIN apps a ON d.id = a.developer_id
        GROUP BY d.id
        ORDER BY d.created_at DESC
    ");
    $stmt->execute();
    $developers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode($developers);
}

function handleCreateDeveloper() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        INSERT INTO developers (name, email, website, bio, verified, created_at) 
        VALUES (?, ?, ?, ?, ?, NOW())
    ");
    
    $stmt->execute([
        $_POST['name'],
        $_POST['email'],
        $_POST['website'] ?: null,
        $_POST['bio'] ?: null,
        isset($_POST['verified']) ? 1 : 0
    ]);
    
    echo json_encode(['success' => true, 'id' => $pdo->lastInsertId()]);
}

function handleUpdateDeveloper() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        UPDATE developers SET 
            name = ?, email = ?, website = ?, bio = ?, verified = ?
        WHERE id = ?
    ");
    
    $stmt->execute([
        $_POST['name'],
        $_POST['email'],
        $_POST['website'] ?: null,
        $_POST['bio'] ?: null,
        isset($_POST['verified']) ? 1 : 0,
        intval($_POST['id'])
    ]);
    
    echo json_encode(['success' => true]);
}

function handleDeleteDeveloper() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $id = intval($input['id']);
    
    $stmt = $pdo->prepare("DELETE FROM developers WHERE id = ?");
    $stmt->execute([$id]);
    
    echo json_encode(['success' => true]);
}

function handleGetStats() {
    global $pdo;
    
    $totalApps = $pdo->query("SELECT COUNT(*) FROM apps")->fetchColumn();
    $activeApps = $pdo->query("SELECT COUNT(*) FROM apps WHERE status = 'active'")->fetchColumn();
    $totalDownloads = $pdo->query("SELECT SUM(downloads) FROM apps")->fetchColumn();
    $featuredApps = $pdo->query("SELECT COUNT(*) FROM apps WHERE featured = 1")->fetchColumn();
    
    $categoryStats = $pdo->query("
        SELECT 
            category,
            COUNT(*) as count,
            SUM(downloads) as downloads
        FROM apps 
        GROUP BY category
        ORDER BY count DESC
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $recentApps = $pdo->query("
        SELECT name, created_at, downloads, category
        FROM apps 
        ORDER BY created_at DESC 
        LIMIT 5
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'totals' => [
            'apps' => $totalApps,
            'active_apps' => $activeApps,
            'total_downloads' => $totalDownloads,
            'featured_apps' => $featuredApps
        ],
        'categories' => $categoryStats,
        'recent_apps' => $recentApps
    ]);
}

function uploadFile($file, $subfolder) {
    $uploadDir = __DIR__ . '/upload-content/' . $subfolder . '/';
    $webPath = '/upload-content/' . $subfolder . '/';
    
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $fileName = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file['name']);
    $filePath = $uploadDir . $fileName;
    
    if (!move_uploaded_file($file['tmp_name'], $filePath)) {
        throw new Exception('Failed to upload file');
    }
    
    return 'https://store.imators.com/admin' . $webPath . $fileName;
}

function deleteFile($url) {
    if (strpos($url, 'store.imators.com') !== false) {
        $path = parse_url($url, PHP_URL_PATH);
        $filePath = __DIR__ . $path;
        if (file_exists($filePath)) {
            unlink($filePath);
        }
    }
}