<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$type = $_POST['type'] ?? 'general';
$allowedTypes = ['icons', 'downloads', 'screenshots', 'general'];

if (!in_array($type, $allowedTypes)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid upload type']);
    exit;
}

if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    echo json_encode(['error' => 'No file uploaded or upload error']);
    exit;
}

$file = $_FILES['file'];

$maxSizes = [
    'icons' => 2 * 1024 * 1024,        // 2MB for icons
    'downloads' => 500 * 1024 * 1024,  // 500MB for downloads
    'screenshots' => 5 * 1024 * 1024,  // 5MB for screenshots
    'general' => 10 * 1024 * 1024      // 10MB for general files
];

$allowedMimeTypes = [
    'icons' => ['image/png', 'image/jpeg', 'image/jpg', 'image/webp'],
    'downloads' => ['application/zip', 'application/x-zip-compressed', 'application/octet-stream', 'application/x-executable'],
    'screenshots' => ['image/png', 'image/jpeg', 'image/jpg', 'image/webp'],
    'general' => ['image/png', 'image/jpeg', 'image/jpg', 'image/webp', 'application/zip', 'application/pdf']
];

if ($file['size'] > $maxSizes[$type]) {
    http_response_code(400);
    echo json_encode(['error' => 'File too large']);
    exit;
}

$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mimeType = finfo_file($finfo, $file['tmp_name']);
finfo_close($finfo);

if (!in_array($mimeType, $allowedMimeTypes[$type])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid file type']);
    exit;
}

try {
    $uploadDir = __DIR__ . '/upload-content/' . $type . '/';
    $webPath = '/upload-content/' . $type . '/';
    
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $fileName = time() . '_' . uniqid() . '.' . $fileExtension;
    $filePath = $uploadDir . $fileName;
    
    if (!move_uploaded_file($file['tmp_name'], $filePath)) {
        throw new Exception('Failed to move uploaded file');
    }
    
    chmod($filePath, 0644);
    
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $fullUrl = $protocol . '://' . $host . $webPath . $fileName;
    
    echo json_encode([
        'success' => true,
        'url' => $fullUrl,
        'filename' => $fileName,
        'size' => $file['size'],
        'type' => $mimeType
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Upload failed: ' . $e->getMessage()]);
}