<?php
require_once 'db.php';
require_once 'vendor/autoload.php';

use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Transfer;
use Stripe\Account;

class StripePaymentHandler {
    
    public function __construct() {
        $keys = getStripeKeys();
        Stripe::setApiKey($keys['secret']);
    }
    
    public function createPaymentIntent($amount, $currency = 'eur', $metadata = []) {
        try {
            $paymentIntent = PaymentIntent::create([
                'amount' => $amount * 100,
                'currency' => $currency,
                'metadata' => $metadata,
                'payment_method_types' => ['card'],
                'capture_method' => 'manual'
            ]);
            
            return [
                'success' => true,
                'client_secret' => $paymentIntent->client_secret,
                'payment_intent_id' => $paymentIntent->id
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function confirmPayment($paymentIntentId) {
        try {
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);
            $paymentIntent->confirm();
            
            return [
                'success' => true,
                'status' => $paymentIntent->status
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function capturePayment($paymentIntentId, $amountToCapture = null) {
        try {
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);
            
            $captureData = [];
            if ($amountToCapture) {
                $captureData['amount_to_capture'] = $amountToCapture * 100;
            }
            
            $paymentIntent->capture($captureData);
            
            return [
                'success' => true,
                'status' => $paymentIntent->status
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function transferToDriver($amount, $driverAccountId, $transferGroup = null) {
        try {
            $transfer = Transfer::create([
                'amount' => $amount * 100,
                'currency' => 'eur',
                'destination' => $driverAccountId,
                'transfer_group' => $transferGroup,
                'metadata' => [
                    'type' => 'delivery_fee',
                    'description' => 'Frais de livraison - Vert Chasseur'
                ]
            ]);
            
            return [
                'success' => true,
                'transfer_id' => $transfer->id
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function createConnectedAccount($email, $firstName, $lastName, $phone = null) {
        try {
            $account = Account::create([
                'type' => 'express',
                'country' => 'BE',
                'email' => $email,
                'capabilities' => [
                    'card_payments' => ['requested' => true],
                    'transfers' => ['requested' => true],
                ],
                'business_type' => 'individual',
                'individual' => [
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'email' => $email,
                    'phone' => $phone
                ],
                'metadata' => [
                    'platform' => 'vert_chasseur'
                ]
            ]);
            
            return [
                'success' => true,
                'account_id' => $account->id
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function createAccountLink($accountId, $returnUrl, $refreshUrl) {
        try {
            $accountLink = \Stripe\AccountLink::create([
                'account' => $accountId,
                'refresh_url' => $refreshUrl,
                'return_url' => $returnUrl,
                'type' => 'account_onboarding',
            ]);
            
            return [
                'success' => true,
                'url' => $accountLink->url
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function refundPayment($paymentIntentId, $amount = null, $reason = null) {
        try {
            $refundData = [
                'payment_intent' => $paymentIntentId
            ];
            
            if ($amount) {
                $refundData['amount'] = $amount * 100;
            }
            
            if ($reason) {
                $refundData['reason'] = $reason;
            }
            
            $refund = \Stripe\Refund::create($refundData);
            
            return [
                'success' => true,
                'refund_id' => $refund->id,
                'status' => $refund->status
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        http_response_code(403);
        echo json_encode(['error' => 'Token CSRF invalide']);
        exit;
    }
    
    $stripeHandler = new StripePaymentHandler();
    
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create_payment_intent':
                $commandeId = (int)$_POST['commande_id'];
                $commande = db()->fetchOne("SELECT * FROM vr_commandes WHERE id = ?", [$commandeId]);
                
                if (!$commande) {
                    echo json_encode(['error' => 'Commande introuvable']);
                    exit;
                }
                
                $result = $stripeHandler->createPaymentIntent(
                    $commande['total_commande'],
                    'eur',
                    [
                        'commande_id' => $commandeId,
                        'client_name' => $commande['first_name'] . ' ' . $commande['last_name'],
                        'commerce' => $commande['commerce_nom']
                    ]
                );
                
                if ($result['success']) {
                    db()->update('vr_commandes', 
                        ['stripe_payment_intent_id' => $result['payment_intent_id']], 
                        'id = ?', 
                        [$commandeId]
                    );
                }
                
                echo json_encode($result);
                exit;
                
            case 'confirm_payment':
                $paymentIntentId = sanitizeInput($_POST['payment_intent_id']);
                $commandeId = (int)$_POST['commande_id'];
                
                $result = $stripeHandler->confirmPayment($paymentIntentId);
                
                if ($result['success'] && $result['status'] === 'succeeded') {
                    db()->update('vr_commandes', 
                        ['statut' => 'confirmee', 'date_confirmation' => date('Y-m-d H:i:s')], 
                        'id = ?', 
                        [$commandeId]
                    );
                    
                    sendOrderNotificationToDriver($commandeId);
                }
                
                echo json_encode($result);
                exit;
                
            case 'process_delivery_payment':
                $commandeId = (int)$_POST['commande_id'];
                $commande = db()->fetchOne("
                    SELECT c.*, l.stripe_account_id 
                    FROM vr_commandes c 
                    JOIN vr_livreurs l ON c.livreur_id = l.id 
                    WHERE c.id = ?
                ", [$commandeId]);
                
                if (!$commande || !$commande['stripe_account_id']) {
                    echo json_encode(['error' => 'Compte Stripe du livreur non configuré']);
                    exit;
                }
                
                sleep(120);
                
                $result = $stripeHandler->transferToDriver(
                    $commande['frais_livraison'],
                    $commande['stripe_account_id'],
                    'commande_' . $commandeId
                );
                
                if ($result['success']) {
                    db()->update('vr_commandes', 
                        ['stripe_transfer_id' => $result['transfer_id']], 
                        'id = ?', 
                        [$commandeId]
                    );
                }
                
                echo json_encode($result);
                exit;
                
            case 'create_driver_account':
                $livreurId = (int)$_POST['livreur_id'];
                $livreur = db()->fetchOne("SELECT * FROM vr_livreurs WHERE id = ?", [$livreurId]);
                
                if (!$livreur) {
                    echo json_encode(['error' => 'Livreur introuvable']);
                    exit;
                }
                
                $result = $stripeHandler->createConnectedAccount(
                    $livreur['email'],
                    $livreur['prenom'],
                    $livreur['nom'],
                    $livreur['telephone']
                );
                
                if ($result['success']) {
                    db()->update('vr_livreurs', 
                        ['stripe_account_id' => $result['account_id']], 
                        'id = ?', 
                        [$livreurId]
                    );
                    
                    $linkResult = $stripeHandler->createAccountLink(
                        $result['account_id'],
                        'https://vertchasseur.com/livreur.php?setup=success',
                        'https://vertchasseur.com/livreur.php?setup=refresh'
                    );
                    
                    if ($linkResult['success']) {
                        $result['onboarding_url'] = $linkResult['url'];
                    }
                }
                
                echo json_encode($result);
                exit;
                
            case 'refund_order':
                $commandeId = (int)$_POST['commande_id'];
                $reason = sanitizeInput($_POST['reason'] ?? '');
                
                $commande = db()->fetchOne("SELECT * FROM vr_commandes WHERE id = ?", [$commandeId]);
                
                if (!$commande || !$commande['stripe_payment_intent_id']) {
                    echo json_encode(['error' => 'Commande ou paiement introuvable']);
                    exit;
                }
                
                $result = $stripeHandler->refundPayment(
                    $commande['stripe_payment_intent_id'],
                    null,
                    $reason
                );
                
                if ($result['success']) {
                    db()->update('vr_commandes', 
                        ['statut' => 'annulee'], 
                        'id = ?', 
                        [$commandeId]
                    );
                }
                
                echo json_encode($result);
                exit;
        }
    }
}

function sendOrderNotificationToDriver($commandeId) {
    $commande = db()->fetchOne("
        SELECT c.*, l.prenom, l.nom, r.first_name, r.last_name 
        FROM vr_commandes c 
        JOIN vr_livreurs l ON c.livreur_id = l.id 
        JOIN residents r ON c.resident_id = r.id 
        WHERE c.id = ?
    ", [$commandeId]);
    
    $oneSignalAppId = db()->getConfig('onesignal_app_id');
    $oneSignalApiKey = db()->getConfig('onesignal_api_key');
    
    if ($oneSignalAppId && $oneSignalApiKey) {
        $notificationData = [
            'app_id' => $oneSignalAppId,
            'filters' => [
                ['field' => 'tag', 'key' => 'livreur_id', 'relation' => '=', 'value' => $commande['livreur_id']]
            ],
            'headings' => ['fr' => 'Nouvelle commande !'],
            'contents' => ['fr' => "Commande #{$commandeId} de {$commande['first_name']} {$commande['last_name']} - {$commande['total_commande']}€"],
            'data' => [
                'type' => 'new_order',
                'commande_id' => $commandeId
            ]
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://onesignal.com/api/v1/notifications');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Basic ' . $oneSignalApiKey
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($notificationData));
        
        curl_exec($ch);
        curl_close($ch);
    }
}

function processDelayedDeliveryPayment($commandeId) {
    $stripeHandler = new StripePaymentHandler();
    $commande = db()->fetchOne("
        SELECT c.*, l.stripe_account_id 
        FROM vr_commandes c 
        JOIN vr_livreurs l ON c.livreur_id = l.id 
        WHERE c.id = ? AND c.statut = 'livree' AND c.stripe_transfer_id IS NULL
    ", [$commandeId]);
    
    if ($commande && $commande['stripe_account_id']) {
        $result = $stripeHandler->transferToDriver(
            $commande['frais_livraison'],
            $commande['stripe_account_id'],
            'commande_' . $commandeId
        );
        
        if ($result['success']) {
            db()->update('vr_commandes', 
                ['stripe_transfer_id' => $result['transfer_id']], 
                'id = ?', 
                [$commandeId]
            );
        }
    }
}
?>