<?php
session_start();
require_once __DIR__ . '/vendor/autoload.php';
require_once 'db.php';

$stripe_config = require 'config/stripe.php';
\Stripe\Stripe::setApiKey($stripe_config['secret_key']);

if (!isset($_SESSION['resident_id'])) {
    header('Location: login.php');
    exit;
}

$resident_id = $_SESSION['resident_id'];

$stmt = $conn->prepare("SELECT * FROM residents WHERE id = ?");
$stmt->bind_param("i", $resident_id);
$stmt->execute();
$resident = $stmt->get_result()->fetch_assoc();

$sub_stmt = $conn->prepare("SELECT * FROM subscriptions WHERE resident_id = ? ORDER BY created_at DESC LIMIT 1");
$sub_stmt->bind_param("i", $resident_id);
$sub_stmt->execute();
$subscription = $sub_stmt->get_result()->fetch_assoc();

if (!$subscription) {
    header('Location: subscription.php');
    exit;
}

$payments_stmt = $conn->prepare("SELECT * FROM payment_history WHERE resident_id = ? ORDER BY payment_date DESC LIMIT 10");
$payments_stmt->bind_param("i", $resident_id);
$payments_stmt->execute();
$payments = $payments_stmt->get_result();

$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action'])) {
            if ($_POST['action'] === 'cancel_subscription') {
                $stripe_subscription = \Stripe\Subscription::retrieve($subscription['stripe_subscription_id']);
                $stripe_subscription->cancel();
                
                $update = $conn->prepare("UPDATE subscriptions SET status = 'canceled', cancel_at_period_end = 1 WHERE id = ?");
                $update->bind_param("i", $subscription['id']);
                $update->execute();
                
                $update_member = $conn->prepare("UPDATE residents SET member_ = 0 WHERE id = ?");
                $update_member->bind_param("i", $resident_id);
                $update_member->execute();
                
                $success_message = 'Abonnement annulé avec succès.';
                header('Location: manage-subscription.php?canceled=1');
                exit;
            }
            
            if ($_POST['action'] === 'reactivate_subscription') {
                $stripe_subscription = \Stripe\Subscription::retrieve($subscription['stripe_subscription_id']);
                \Stripe\Subscription::update($subscription['stripe_subscription_id'], [
                    'cancel_at_period_end' => false
                ]);
                
                $update = $conn->prepare("UPDATE subscriptions SET cancel_at_period_end = 0 WHERE id = ?");
                $update->bind_param("i", $subscription['id']);
                $update->execute();
                
                $success_message = 'Abonnement réactivé avec succès.';
                header('Location: manage-subscription.php?reactivated=1');
                exit;
            }
        }
    } catch (Exception $e) {
        $error_message = 'Erreur : ' . $e->getMessage();
    }
}

$status_labels = [
    'active' => ['text' => 'Actif', 'class' => 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300'],
    'canceled' => ['text' => 'Annulé', 'class' => 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-300'],
    'past_due' => ['text' => 'Paiement en retard', 'class' => 'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-300'],
    'unpaid' => ['text' => 'Non payé', 'class' => 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-300'],
    'trialing' => ['text' => 'Période d\'essai', 'class' => 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-300']
];

$current_status = $status_labels[$subscription['status']] ?? ['text' => $subscription['status'], 'class' => 'bg-gray-100 text-gray-800'];
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gérer mon abonnement - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {}
            }
        }
    </script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            letter-spacing: -0.01em;
        }
    </style>
</head>

<body class="bg-white dark:bg-black transition-colors duration-300">
    <script>
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    </script>

    <div class="min-h-screen py-8 md:py-12 px-4">
        <div class="max-w-4xl mx-auto">
            <div class="mb-8">
                <a href="/resident-dashboard" class="text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 text-sm inline-flex items-center gap-2 mb-4 transition-colors">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                    </svg>
                    Retour
                </a>
                <h1 class="text-3xl md:text-4xl font-light text-stone-800 dark:text-white tracking-tight">Gérer mon abonnement</h1>
                <p class="text-stone-600 dark:text-stone-400 mt-2">Consulte et modifie ton abonnement</p>
            </div>

            <?php if (isset($_GET['canceled'])): ?>
            <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-xl p-4 mb-6">
                <p class="text-green-800 dark:text-green-200">Ton abonnement a été annulé avec succès.</p>
            </div>
            <?php endif; ?>

            <?php if (isset($_GET['reactivated'])): ?>
            <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-xl p-4 mb-6">
                <p class="text-green-800 dark:text-green-200">Ton abonnement a été réactivé avec succès.</p>
            </div>
            <?php endif; ?>

            <?php if ($error_message): ?>
            <div class="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-xl p-4 mb-6">
                <p class="text-red-800 dark:text-red-200"><?= htmlspecialchars($error_message) ?></p>
            </div>
            <?php endif; ?>

            <div class="bg-white dark:bg-stone-900 rounded-2xl md:rounded-3xl p-6 md:p-8 border border-stone-200 dark:border-stone-800 shadow-lg dark:shadow-stone-900/30 mb-6">
                <h2 class="text-xl md:text-2xl font-light text-stone-800 dark:text-white mb-6 tracking-tight">Détails de l'abonnement</h2>
                
                <div class="space-y-4">
                    <div class="flex justify-between items-center py-3 border-b border-stone-200 dark:border-stone-700">
                        <span class="text-stone-600 dark:text-stone-400">Statut</span>
                        <span class="px-3 py-1 rounded-full text-sm font-medium <?= $current_status['class'] ?>">
                            <?= $current_status['text'] ?>
                        </span>
                    </div>

                    <div class="flex justify-between items-center py-3 border-b border-stone-200 dark:border-stone-700">
                        <span class="text-stone-600 dark:text-stone-400">Prix</span>
                        <span class="text-stone-900 dark:text-white font-semibold">1,99€ / mois</span>
                    </div>

                    <div class="flex justify-between items-center py-3 border-b border-stone-200 dark:border-stone-700">
                        <span class="text-stone-600 dark:text-stone-400">Début de période</span>
                        <span class="text-stone-900 dark:text-white"><?= date('d/m/Y', strtotime($subscription['current_period_start'])) ?></span>
                    </div>

                    <div class="flex justify-between items-center py-3 border-b border-stone-200 dark:border-stone-700">
                        <span class="text-stone-600 dark:text-stone-400">Fin de période</span>
                        <span class="text-stone-900 dark:text-white"><?= date('d/m/Y', strtotime($subscription['current_period_end'])) ?></span>
                    </div>

                    <div class="flex justify-between items-center py-3">
                        <span class="text-stone-600 dark:text-stone-400">Renouvellement automatique</span>
                        <span class="text-stone-900 dark:text-white">
                            <?= $subscription['cancel_at_period_end'] ? 'Non' : 'Oui' ?>
                        </span>
                    </div>
                </div>

                <?php if ($subscription['status'] === 'active' && !$subscription['cancel_at_period_end']): ?>
                <form method="POST" onsubmit="return confirm('Es-tu sûr de vouloir annuler ton abonnement ?')" class="mt-8">
                    <input type="hidden" name="action" value="cancel_subscription">
                    <button type="submit" class="w-full bg-red-600 hover:bg-red-700 text-white font-semibold py-3 rounded-xl transition-colors duration-200">
                        Annuler l'abonnement
                    </button>
                    <p class="text-xs text-center text-stone-500 dark:text-stone-400 mt-2">
                        Tu conserveras l'accès jusqu'au <?= date('d/m/Y', strtotime($subscription['current_period_end'])) ?>
                    </p>
                </form>
                <?php elseif ($subscription['cancel_at_period_end']): ?>
                <form method="POST" class="mt-8">
                    <input type="hidden" name="action" value="reactivate_subscription">
                    <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 rounded-xl transition-colors duration-200">
                        Réactiver l'abonnement
                    </button>
                    <p class="text-xs text-center text-stone-500 dark:text-stone-400 mt-2">
                        Ton abonnement sera automatiquement renouvelé
                    </p>
                </form>
                <?php endif; ?>
            </div>

            <div class="bg-white dark:bg-stone-900 rounded-2xl md:rounded-3xl p-6 md:p-8 border border-stone-200 dark:border-stone-800 shadow-lg dark:shadow-stone-900/30">
                <h2 class="text-xl md:text-2xl font-light text-stone-800 dark:text-white mb-6 tracking-tight">Historique des paiements</h2>
                
                <?php if ($payments->num_rows > 0): ?>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead>
                            <tr class="border-b border-stone-200 dark:border-stone-700">
                                <th class="text-left py-3 px-4 text-stone-600 dark:text-stone-400 font-medium text-sm">Date</th>
                                <th class="text-left py-3 px-4 text-stone-600 dark:text-stone-400 font-medium text-sm">Montant</th>
                                <th class="text-left py-3 px-4 text-stone-600 dark:text-stone-400 font-medium text-sm">Statut</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while($payment = $payments->fetch_assoc()): ?>
                            <tr class="border-b border-stone-100 dark:border-stone-800">
                                <td class="py-4 px-4 text-stone-900 dark:text-white">
                                    <?= date('d/m/Y H:i', strtotime($payment['payment_date'])) ?>
                                </td>
                                <td class="py-4 px-4 text-stone-900 dark:text-white font-semibold">
                                    <?= number_format($payment['amount'], 2, ',', ' ') ?>€
                                </td>
                                <td class="py-4 px-4">
                                    <?php if ($payment['status'] === 'succeeded' || $payment['status'] === 'paid'): ?>
                                    <span class="px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300">
                                        Payé
                                    </span>
                                    <?php else: ?>
                                    <span class="px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-300">
                                        <?= htmlspecialchars($payment['status']) ?>
                                    </span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <p class="text-center text-stone-500 dark:text-stone-400 py-8">
                    Aucun paiement enregistré
                </p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>