<?php
include 'db.php';
include 'header-mobile.php';

function parseShopHours($hours, $days) {
    list($openHour, $closeHour) = explode(' à ', $hours);
    $daysArray = explode(',', $days);
    $daysArray = array_map('trim', $daysArray);

    $isMorningOpen = strtotime($openHour) <= strtotime('10:00');
    $isLateOpen = strtotime($closeHour) >= strtotime('20:00');
    $isWeekendOpen = in_array('Saturday', $daysArray) || in_array('Sunday', $daysArray);

    $schedule = [];
    $allDays = ['Monday','Tuesday','Wednesday','Thursday','Friday','Saturday','Sunday'];
    $dayTranslation = [
        'Monday' => 'Lundi',
        'Tuesday' => 'Mardi',
        'Wednesday' => 'Mercredi',
        'Thursday' => 'Jeudi',
        'Friday' => 'Vendredi',
        'Saturday' => 'Samedi',
        'Sunday' => 'Dimanche'
    ];

    foreach ($allDays as $day) {
        if (in_array($day, $daysArray)) {
            $schedule[$dayTranslation[$day]] = [$openHour . ' - ' . $closeHour];
        } else {
            $schedule[$dayTranslation[$day]] = ['Fermé'];
        }
    }

    return [
        'open_hour' => $openHour,
        'close_hour' => $closeHour,
        'open_days' => $daysArray,
        'morning_open' => $isMorningOpen,
        'late_open' => $isLateOpen,
        'weekend_open' => $isWeekendOpen,
        'formatted_schedule' => $schedule
    ];
}

$shops = mysqli_query($conn, "SELECT * FROM shop WHERE active = 1 ORDER BY price_smilemore DESC, title ASC");
$shopsList = [];
$shopsByCategory = [];

while ($shop = mysqli_fetch_assoc($shops)) {
    $category = $shop['category'];
    if (!isset($shopsByCategory[$category])) {
        $shopsByCategory[$category] = [];
    }
    $shopsByCategory[$category][] = $shop;
    $shopsList[] = $shop;
}

$shopsJson = json_encode($shopsList);

$zoneData = [
    'center' => ['lat' => 50.800740898699665, 'lng' => 4.370884895324708, 'zoom' => 16],
    'zone' => [
        ['lat' => 50.80547367922133, 'lng' => 4.376249313354493],
        ['lat' => 50.801961432385966, 'lng' => 4.377751350402833],
        ['lat' => 50.79931690243816, 'lng' => 4.378888607025147],
        ['lat' => 50.796712911337075, 'lng' => 4.379446506500245],
        ['lat' => 50.79622464685215, 'lng' => 4.374210834503175],
        ['lat' => 50.79600763877701, 'lng' => 4.367022514343263],
        ['lat' => 50.797282546788885, 'lng' => 4.366786479949952],
        ['lat' => 50.79747242372991, 'lng' => 4.365756511688233],
        ['lat' => 50.79846248385208, 'lng' => 4.364705085754395],
        ['lat' => 50.79934402658185, 'lng' => 4.363932609558106],
        ['lat' => 50.801134185257695, 'lng' => 4.362559318542481],
        ['lat' => 50.80212416780201, 'lng' => 4.362301826477052],
        ['lat' => 50.80304632445409, 'lng' => 4.362366199493409],
        ['lat' => 50.80373792999973, 'lng' => 4.3625807762146005],
        ['lat' => 50.80455157047645, 'lng' => 4.367430210113526],
        ['lat' => 50.80506686878319, 'lng' => 4.370605945587159],
        ['lat' => 50.80532451580529, 'lng' => 4.372129440307618],
        ['lat' => 50.80546011893046, 'lng' => 4.373846054077149]
    ]
];
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Commerces - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <meta name="description" content="Découvrez tous les commerces et restaurants partenaires dans le quartier de Vert Chasseur à Uccle, Bruxelles.">
    
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: { extend: {} }
        }
    </script>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            letter-spacing: -0.01em;
        }

        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        @keyframes scaleIn {
            from {
                opacity: 0;
                transform: scale(0.95);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }

        .animate-fade-in-up {
            animation: fadeInUp 0.8s ease-out forwards;
            opacity: 0;
        }

        .animate-scale-in {
            animation: scaleIn 0.6s ease-out forwards;
            opacity: 0;
        }

        .card {
            transition: all 0.4s cubic-bezier(0.165, 0.84, 0.44, 1);
            cursor: pointer;
            opacity: 0;
            animation: fadeInUp 0.6s ease-out forwards;
        }

        .card:hover {
            transform: translateY(-12px);
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
        }

        .dark .card:hover {
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.5);
        }

        .status-dot {
            animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
        }

        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }

        .shop-image {
            transition: transform 0.8s cubic-bezier(0.165, 0.84, 0.44, 1);
        }

        .card:hover .shop-image {
            transform: scale(1.08);
        }

        .category-scroll {
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .category-scroll::-webkit-scrollbar {
            display: none;
        }

        .map-container {
            position: relative;
            width: 100%;
            height: 450px;
            border-radius: 20px;
            overflow: hidden;
            opacity: 0;
            transform: scale(0.95);
            transition: all 0.6s cubic-bezier(0.165, 0.84, 0.44, 1);
        }

        .map-container.visible {
            opacity: 1;
            transform: scale(1);
        }

        .map-container.hidden {
            display: none;
        }

        #map {
            width: 100%;
            height: 100%;
        }

        .location-dot {
            width: 22px;
            height: 22px;
            background: linear-gradient(135deg, #0ea5e9 0%, #06b6d4 100%);
            border-radius: 50%;
            border: 4px solid white;
            box-shadow: 0 0 0 3px rgba(14, 165, 233, 0.4);
            animation: locationPulse 2.5s infinite;
        }

        @keyframes locationPulse {
            0% { box-shadow: 0 0 0 0 rgba(14, 165, 233, 0.8); }
            70% { box-shadow: 0 0 0 20px rgba(14, 165, 233, 0); }
            100% { box-shadow: 0 0 0 0 rgba(14, 165, 233, 0); }
        }

        .blip-icon {
            width: 48px;
            height: 48px;
            background: white;
            border: 3px solid #0ea5e9;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            color: #0ea5e9;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.15);
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.165, 0.84, 0.44, 1);
        }

        .blip-icon:hover {
            transform: scale(1.2);
            box-shadow: 0 8px 24px rgba(14, 165, 233, 0.4);
        }

        .dark .blip-icon {
            background: #1a1a1a;
            border-color: #0ea5e9;
        }

        .leaflet-popup-content-wrapper {
            background: white;
            border-radius: 16px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.15);
            border: none;
            color: black;
        }

        .dark .leaflet-popup-content-wrapper {
            background: #1a1a1a;
            color: white;
        }

        .dark .leaflet-tile {
            filter: invert(0.93) hue-rotate(180deg) brightness(1.05) contrast(0.9);
        }

        .leaflet-control-container {
            display: none;
        }

        @media (max-width: 768px) {
            .map-container {
                height: 380px;
            }
        }

        .closing-soon-badge {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            animation: urgentPulse 1.5s ease-in-out infinite;
        }

        @keyframes urgentPulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }

        .filter-overlay {
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0);
            z-index: 50;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .filter-overlay.active {
            visibility: visible;
            background: rgba(0, 0, 0, 0.5);
        }

        .dark .filter-overlay.active {
            background: rgba(0, 0, 0, 0.5);
        }

        .filter-overlay:not(.dark).active {
            background: rgba(255, 255, 255, 0.8);
        }

        .filter-panel {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            border-radius: 24px 24px 0 0;
            transform: translateY(100%);
            transition: transform 0.3s cubic-bezier(0.165, 0.84, 0.44, 1);
            max-height: 80vh;
            overflow-y: auto;
            z-index: 51;
        }

        .dark .filter-panel {
            background: #1a1a1a;
        }

        .filter-overlay.active .filter-panel {
            transform: translateY(0);
        }

        .filter-count {
            position: absolute;
            top: -5px;
            right: -5px;
            background: #ef4444;
            color: white;
            width: 20px;
            height: 20px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 11px;
            font-weight: 600;
        }

        @keyframes slideInFromBottom {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .filter-item {
            animation: slideInFromBottom 0.3s ease-out forwards;
            opacity: 0;
        }

        @media (max-width: 640px) {
            .search-filter-wrapper {
                flex-direction: column;
            }
            
            .search-filter-wrapper > div,
            .search-filter-wrapper > button {
                width: 100%;
            }
        }
    </style>
</head>

<body class="bg-white dark:bg-black text-black dark:text-white transition-colors duration-300">
    <script>
        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            document.documentElement.classList.add('dark');
        }
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', e => {
            if (e.matches) {
                document.documentElement.classList.add('dark');
            } else {
                document.documentElement.classList.remove('dark');
            }
        });
    </script>

    <main class="px-4 py-8 md:py-16 max-w-6xl mx-auto">
        <div class="flex flex-col items-center justify-center mb-8 md:mb-12 text-center relative">

            <h1 class="mt-4 md:mt-12 text-3xl md:text-5xl lg:text-6xl font-light text-stone-800 dark:text-white mb-2 md:mb-3 tracking-tight">
                Commerces
            </h1>
            <p class="text-lg md:text-xl text-stone-600 dark:text-stone-400 font-light">
                Découvrez les enseignes qui font la richesse du quartier.
            </p>
        </div>

        <div id="mapSection" class="mb-16 map-container hidden bg-gradient-to-br from-gray-100 to-gray-200 dark:from-stone-900 dark:to-stone-800 shadow-2xl dark:shadow-stone-900/50">
            <div id="map"></div>
        </div>

        <div class="mb-16 animate-fade-in-up" style="animation-delay: 0.5s">
            <div class="search-filter-wrapper flex gap-4 items-center justify-center max-w-3xl mx-auto">
                <div class="relative flex-1">
                    <input type="text"
                           id="shop-search"
                           placeholder="Rechercher un commerce..."
                           class="w-full px-7 py-4 rounded-2xl bg-white dark:bg-stone-900 shadow-sm dark:shadow-stone-900/30 border border-gray-200 dark:border-stone-800 focus:border-gray-400 dark:focus:border-gray-600 focus:outline-none text-black dark:text-white transition-all duration-300">
                    <svg class="absolute right-6 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                    </svg>
                </div>
                
                <button id="filter-toggle" class="relative px-6 py-4 rounded-2xl transition-all duration-300 hover:shadow-lg text-sm font-medium bg-black dark:bg-white text-white dark:text-black whitespace-nowrap">
                    <i class="fas fa-filter mr-2"></i>Filtre de recherche
                    <span id="filter-count" class="filter-count hidden">0</span>
                </button>
            </div>
        </div>

        <div id="filter-overlay" class="filter-overlay">
            <div class="filter-panel">
                <div class="p-6 border-b border-gray-200 dark:border-gray-800">
                    <div class="flex items-center justify-between mb-2">
                        <h3 class="text-xl font-semibold">Filtres</h3>
                        <button id="filter-close" class="w-8 h-8 rounded-full flex items-center justify-center hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <p class="text-sm text-gray-500 dark:text-gray-400">Affinez votre recherche</p>
                </div>

                <div class="p-6 space-y-4">
                    <div class="filter-item" style="animation-delay: 0.05s">
                        <button id="filter-open-now" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-door-open mr-3 text-green-500"></i>Ouvert maintenant</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>

                    <div class="filter-item" style="animation-delay: 0.1s">
                        <button id="filter-last-minute" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-clock mr-3 text-red-500"></i>Dernières minutes</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>

                    <div class="filter-item" style="animation-delay: 0.15s">
                        <button id="filter-no-reservation" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-walking mr-3 text-blue-500"></i>Sans réservation</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>

                    <div class="filter-item" style="animation-delay: 0.2s">
                        <button id="filter-morning" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-sun mr-3 text-yellow-500"></i>Ouvert le matin</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>

                    <div class="filter-item" style="animation-delay: 0.25s">
                        <button id="filter-late" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-moon mr-3 text-indigo-500"></i>Après 20h</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>

                    <div class="filter-item" style="animation-delay: 0.3s">
                        <button id="filter-weekend" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-calendar mr-3 text-purple-500"></i>Week-end</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>

                    <div class="filter-item" style="animation-delay: 0.35s">
                        <button id="filter-delivery" class="filter-button w-full px-5 py-4 rounded-xl transition-all duration-300 text-left font-medium bg-gray-50 dark:bg-stone-900 hover:bg-gray-100 dark:hover:bg-stone-800 flex items-center justify-between">
                            <span><i class="fas fa-truck mr-3 text-orange-500"></i>Livraison privée</span>
                            <i class="fas fa-check opacity-0 transition-opacity duration-200"></i>
                        </button>
                    </div>
                </div>

                <div class="p-6 border-t border-gray-200 dark:border-gray-800">
                    <button id="filter-reset" class="w-full py-3 rounded-xl text-sm font-medium text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">
                        Réinitialiser les filtres
                    </button>
                </div>
            </div>
        </div>

        <?php foreach ($shopsByCategory as $category => $shops): ?>
        <section class="mb-20">
            <div class="flex items-center gap-4 mb-10">
                <span class="text-xs font-bold uppercase tracking-widest text-gray-600 dark:text-gray-500"><?= htmlspecialchars($category) ?></span>
                <div class="h-px flex-1 bg-gradient-to-r from-gray-300 dark:from-gray-800 to-transparent"></div>
            </div>
            
            <div class="category-scroll overflow-x-auto flex gap-7 pb-6">
                <?php
                $delay = 0;
                $firstOnePlaced = false;
                foreach ($shops as $shop):
                    $openingInfo = parseShopHours($shop['hour_open'], $shop['day_open']);
                    $mainImage = trim(explode(',', trim($shop['url_image'], '()'))[0]);
                    $isFirstOne = $shop['first_one'] == 1 && !$firstOnePlaced;
                    if ($isFirstOne) $firstOnePlaced = true;
                ?>
                <article class="card bg-white dark:bg-stone-900 rounded-3xl overflow-hidden border border-gray-100 dark:border-stone-800 shadow-xl dark:shadow-stone-900/40 flex-shrink-0 w-80 relative"
                         data-shop
                         data-shop-id="<?= $shop['id'] ?>"
                         data-open-hour="<?= htmlspecialchars($openingInfo['open_hour']) ?>"
                         data-close-hour="<?= htmlspecialchars($openingInfo['close_hour']) ?>"
                         data-open-days='<?= json_encode($openingInfo['open_days']) ?>'
                         data-title="<?= htmlspecialchars($shop['title']) ?>"
                         data-description="<?= htmlspecialchars($shop['description']) ?>"
                         data-morning="<?= $openingInfo['morning_open'] ? 'true' : 'false' ?>"
                         data-late="<?= $openingInfo['late_open'] ? 'true' : 'false' ?>"
                         data-weekend="<?= $openingInfo['weekend_open'] ? 'true' : 'false' ?>"
                         data-delivery="<?= $shop['delivery_possible'] == 1 ? 'true' : 'false' ?>"
                         data-no-reservation="<?= $shop['reservation_taked'] == 0 ? 'true' : 'false' ?>"
                         data-category="<?= htmlspecialchars($shop['category']) ?>"
                         style="animation-delay: <?= $delay ?>s; order: <?= $isFirstOne ? '-1' : '0' ?>">
                    
                    <div class="aspect-[4/3] relative overflow-hidden">
                        <img src="<?= htmlspecialchars($mainImage) ?>"
                             alt="<?= htmlspecialchars($shop['title']) ?>"
                             class="w-full h-full object-cover shop-image">

                        <div class="absolute inset-0 bg-gradient-to-t from-black/70 via-black/30 to-transparent"></div>

                        <?php if($shop['price_smilemore'] == 1): ?>
                        <div class="absolute top-5 right-5 bg-white/15 backdrop-blur-xl px-4 py-2 rounded-full text-sm font-medium text-white border border-white/30 shadow-lg">
                            <i class="fas fa-award mr-2"></i>Populaire
                        </div>
                        <?php endif; ?>

                        <div class="absolute top-5 left-5">
                            <span class="status-badge inline-flex items-center px-4 py-2 rounded-full text-xs font-semibold bg-red-500 text-white shadow-lg">
                                <span class="status-dot w-2 h-2 rounded-full bg-white mr-2"></span>
                                <span class="status-text">Fermé</span>
                            </span>
                        </div>

                        <div class="closing-soon-indicator absolute top-5 right-5 hidden">
                            <span class="closing-soon-badge inline-flex items-center px-4 py-2 rounded-full text-xs font-bold text-white shadow-lg">
                                <i class="fas fa-exclamation-triangle mr-2"></i>Ferme bientôt
                            </span>
                        </div>

                        <div class="absolute bottom-5 left-5 flex gap-2.5">
                            <?php if($openingInfo['morning_open']): ?>
                                <span class="px-3 py-1.5 bg-blue-500/80 text-white backdrop-blur-xl rounded-full text-xs font-medium border border-blue-400/30 shadow-lg">
                                    <i class="fas fa-sun"></i>
                                </span>
                            <?php endif; ?>
                            <?php if($openingInfo['late_open']): ?>
                                <span class="px-3 py-1.5 bg-indigo-500/80 text-white backdrop-blur-xl rounded-full text-xs font-medium border border-indigo-400/30 shadow-lg">
                                    <i class="fas fa-moon"></i>
                                </span>
                            <?php endif; ?>
                            <?php if($openingInfo['weekend_open']): ?>
                                <span class="px-3 py-1.5 bg-purple-500/80 text-white backdrop-blur-xl rounded-full text-xs font-medium border border-purple-400/30 shadow-lg">
                                    <i class="fas fa-calendar"></i>
                                </span>
                            <?php endif; ?>
                        </div>

                        <?php if($shop['parten'] == 1): ?>
                        <div class="absolute bottom-5 right-5 bg-green-500/80 backdrop-blur-xl px-3 py-1.5 rounded-full font-semibold text-white border border-green-400/30 text-xs shadow-lg">
                            -10% membre
                        </div>
                        <?php endif; ?>
                    </div>

                    <div class="p-6">
                        <h3 class="text-xl font-semibold text-black dark:text-white mb-3"><?= htmlspecialchars($shop['title']) ?></h3>

                        <p class="text-sm text-gray-600 dark:text-gray-400 line-clamp-2 mb-5 leading-relaxed"><?= htmlspecialchars($shop['description']) ?></p>

                        <div class="text-sm text-gray-600 dark:text-gray-400 space-y-3 mb-5">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-clock w-4 text-gray-400 dark:text-gray-500"></i>
                                <button class="schedule-button text-left hover:underline text-xs font-medium transition-all" data-shop-id="<?= $shop['id'] ?>">
                                    Voir les horaires
                                </button>
                            </div>

                            <div id="schedule-<?= $shop['id'] ?>" class="mt-3 p-3 bg-gray-50 dark:bg-stone-800 rounded-xl text-xs space-y-1.5" style="display: none;">
                                <?php foreach($openingInfo['formatted_schedule'] as $day => $hours): ?>
                                <div class="flex justify-between">
                                    <span class="font-semibold"><?= $day ?>:</span>
                                    <span>
                                        <?php if (count($hours) === 1 && $hours[0] === 'Fermé'): ?>
                                            <span class="text-red-500 font-medium">Fermé</span>
                                        <?php else: ?>
                                            <?= implode(' & ', $hours) ?>
                                        <?php endif; ?>
                                    </span>
                                </div>
                                <?php endforeach; ?>
                            </div>

                            <div class="next-opening text-xs bg-gray-50 dark:bg-stone-800 p-3 rounded-xl" style="display: none;">
                                <span class="font-semibold">Prochaine ouverture: </span>
                                <span class="next-opening-text"></span>
                            </div>
                        </div>

                        <div class="group block w-full text-center py-3.5 rounded-xl transition-all duration-300 text-sm font-semibold relative overflow-hidden bg-black dark:bg-white text-white dark:text-black hover:shadow-xl hover:scale-105">
                            <span class="flex items-center justify-center gap-2">
                                Plus d'infos
                                <i class="fas fa-arrow-right text-xs transform group-hover:translate-x-1 transition-transform duration-300"></i>
                            </span>
                        </div>
                    </div>
                </article>
                <?php $delay += 0.08; ?>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endforeach; ?>
        
        <div class="text-center mt-12 animate-fade-in-up" style="animation-delay: 0.7s">
            <p class="text-gray-600 dark:text-gray-400 text-sm">
                Ton commerce n'est pas dans la liste ? 
                <a href="https://imators.systems/comm-access" class="text-black dark:text-white font-semibold hover:underline transition-all duration-300">
                    Ajoute-le via Comm Access !
                </a>
            </p>
        </div>
    </main>

    <script>
        function openFilterOverlay() {
            document.getElementById('filter-overlay').classList.add('active');
            document.body.style.overflow = 'hidden';
            
            if (!document.documentElement.classList.contains('dark')) {
                document.getElementById('filter-overlay').classList.remove('dark');
            } else {
                document.getElementById('filter-overlay').classList.add('dark');
            }
        }

        const shopsData = <?= $shopsJson ?>;
        const zoneData = <?= json_encode($zoneData) ?>;
        
        let map = null;
        let userMarker = null;
        let zonePolygon = null;
        let shopMarkers = {};
        let isInZone = false;
        let isRecenteringMap = false;
        let mapInitialized = false;
        
        const targetPos = { lat: null, lng: null };
        const currentPos = { lat: null, lng: null };
        let lastUpdateTime = Date.now();

        const userIcon = L.divIcon({
            className: 'location-dot',
            iconSize: [22, 22],
            iconAnchor: [11, 11]
        });

        function createShopIcon(shop) {
            const icon = shop.category === 'Restaurant' ? 'fa-utensils' : 'fa-store';
            return L.divIcon({
                className: 'blip-icon',
                html: `<i class="fas ${icon}"></i>`,
                iconSize: [48, 48],
                iconAnchor: [24, 24]
            });
        }

        function pointInPolygon(point, polygon) {
            const x = point.lat, y = point.lng;
            let inside = false;
            for (let i = 0, j = polygon.length - 1; i < polygon.length; j = i++) {
                const xi = polygon[i].lat, yi = polygon[i].lng;
                const xj = polygon[j].lat, yj = polygon[j].lng;
                const intersect = ((yi > y) !== (yj > y)) && (x < (xj - xi) * (y - yi) / (yj - yi) + xi);
                if (intersect) inside = !inside;
            }
            return inside;
        }

        function calculateZoneBounds() {
            let minLat = Infinity, maxLat = -Infinity;
            let minLng = Infinity, maxLng = -Infinity;
            
            zoneData.zone.forEach(point => {
                if (point.lat < minLat) minLat = point.lat;
                if (point.lat > maxLat) maxLat = point.lat;
                if (point.lng < minLng) minLng = point.lng;
                if (point.lng > maxLng) maxLng = point.lng;
            });
            
            return { minLat, maxLat, minLng, maxLng };
        }

        const zoneBounds = calculateZoneBounds();

        function updateMapVisibility() {
            const mapSection = document.getElementById('mapSection');
            
            if (isInZone && !mapInitialized) {
                mapSection.classList.remove('hidden');
                setTimeout(() => {
                    mapSection.classList.add('visible');
                    initMap();
                    mapInitialized = true;
                }, 50);
            } else if (!isInZone && mapInitialized) {
                mapSection.classList.remove('visible');
                setTimeout(() => {
                    mapSection.classList.add('hidden');
                }, 600);
            }
        }

        function initMap() {
            map = L.map('map', { 
                zoomControl: false,
                attributionControl: false,
                minZoom: 15,
                maxZoom: 18,
                maxBounds: L.latLngBounds(
                    [zoneBounds.minLat - 0.002, zoneBounds.minLng - 0.002],
                    [zoneBounds.maxLat + 0.002, zoneBounds.maxLng + 0.002]
                ),
                maxBoundsViscosity: 1.0
            }).setView([zoneData.center.lat, zoneData.center.lng], zoneData.center.zoom);

            const isDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
            const tileLayer = isDark 
                ? 'https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png'
                : 'https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png';

            L.tileLayer(tileLayer, { maxZoom: 19 }).addTo(map);

            shopsData.forEach(shop => {
                if (shop.latitude && shop.longitude) {
                    const icon = createShopIcon(shop);
                    const marker = L.marker([parseFloat(shop.latitude), parseFloat(shop.longitude)], { icon: icon }).addTo(map);
                    marker.bindPopup(`<strong>${shop.title}</strong><br><small>${shop.category}</small>`);
                    marker.on('click', () => {
                        window.location.href = 'shop-details-mobile.php?id=' + shop.id;
                    });
                    shopMarkers[shop.id] = marker;
                }
            });

            map.on('movestart', function() {
                isRecenteringMap = false;
            });

            map.on('moveend', function() {
                if (!isRecenteringMap && isInZone && userMarker) {
                    const mapCenter = map.getCenter();
                    const userPos = userMarker.getLatLng();
                    const distance = mapCenter.distanceTo(userPos);
                    
                    if (distance > 100) {
                        isRecenteringMap = true;
                        map.panTo(userPos, { animate: true, duration: 0.5 });
                        setTimeout(() => { isRecenteringMap = false; }, 600);
                    }
                }
            });

            window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
                if (!map) return;
                const newTileLayer = e.matches 
                    ? 'https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png'
                    : 'https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png';
                
                map.eachLayer(layer => {
                    if (layer instanceof L.TileLayer) {
                        map.removeLayer(layer);
                    }
                });
                L.tileLayer(newTileLayer, { maxZoom: 19 }).addTo(map);
            });
        }

        function updateMarker() {
            if (targetPos.lat === null) {
                requestAnimationFrame(updateMarker);
                return;
            }

            if (currentPos.lat === null) {
                currentPos.lat = targetPos.lat;
                currentPos.lng = targetPos.lng;
            }

            const now = Date.now();
            const deltaTime = (now - lastUpdateTime) / 16.67;
            lastUpdateTime = now;

            const smoothFactor = Math.min(0.12 * deltaTime, 1);
            currentPos.lat += (targetPos.lat - currentPos.lat) * smoothFactor;
            currentPos.lng += (targetPos.lng - currentPos.lng) * smoothFactor;

            const wasInZone = isInZone;
            isInZone = pointInPolygon({ lat: currentPos.lat, lng: currentPos.lng }, zoneData.zone);

            if (wasInZone !== isInZone) {
                updateMapVisibility();
            }

            if (isInZone && mapInitialized) {
                if (!userMarker && map) {
                    userMarker = L.marker([currentPos.lat, currentPos.lng], { icon: userIcon }).addTo(map);
                    isRecenteringMap = true;
                    map.setView([currentPos.lat, currentPos.lng], zoneData.center.zoom);
                    setTimeout(() => { isRecenteringMap = false; }, 600);
                } else if (userMarker) {
                    userMarker.setLatLng([currentPos.lat, currentPos.lng]);
                    
                    if (!isRecenteringMap) {
                        const mapCenter = map.getCenter();
                        const userPos = L.latLng(currentPos.lat, currentPos.lng);
                        const distance = mapCenter.distanceTo(userPos);
                        
                        if (distance > 50) {
                            isRecenteringMap = true;
                            map.panTo(userPos, { animate: true, duration: 0.3 });
                            setTimeout(() => { isRecenteringMap = false; }, 400);
                        }
                    }
                }
            }

            requestAnimationFrame(updateMarker);
        }

        navigator.geolocation.watchPosition(
            function(position) {
                targetPos.lat = position.coords.latitude;
                targetPos.lng = position.coords.longitude;
            },
            function(error) {
                console.error('GPS error:', error);
            },
            { enableHighAccuracy: true, maximumAge: 0, timeout: 5000 }
        );

        document.addEventListener('DOMContentLoaded', function() {
            updateMarker();

            let searchInput = document.getElementById('shop-search');
            let filterToggle = document.getElementById('filter-toggle');
            let filterOverlay = document.getElementById('filter-overlay');
            let filterClose = document.getElementById('filter-close');
            let filterReset = document.getElementById('filter-reset');
            let filterCount = document.getElementById('filter-count');
            
            let shopCards = document.querySelectorAll('[data-shop]');
            let currentDay = '';
            let currentTime = '';
            let shopStatus = {};
            let shopClosingSoon = {};
            let nextOpenings = {};
            let activeFilters = new Set();

            function updateFilterCount() {
                const count = activeFilters.size;
                if (count > 0) {
                    filterCount.textContent = count;
                    filterCount.classList.remove('hidden');
                } else {
                    filterCount.classList.add('hidden');
                }
            }

            filterToggle.addEventListener('click', () => {
                filterOverlay.classList.add('active');
                document.body.style.overflow = 'hidden';
                
                if (!document.documentElement.classList.contains('dark')) {
                    filterOverlay.classList.remove('dark');
                } else {
                    filterOverlay.classList.add('dark');
                }
            });

            filterClose.addEventListener('click', () => {
                filterOverlay.classList.remove('active');
                document.body.style.overflow = '';
            });

            filterOverlay.addEventListener('click', (e) => {
                if (e.target === filterOverlay) {
                    filterOverlay.classList.remove('active');
                    document.body.style.overflow = '';
                }
            });

            filterReset.addEventListener('click', () => {
                activeFilters.clear();
                document.querySelectorAll('.filter-button').forEach(btn => {
                    btn.classList.remove('bg-black', 'dark:bg-white', 'text-white', 'dark:text-black');
                    btn.classList.add('bg-gray-50', 'dark:bg-stone-900');
                    btn.querySelector('.fa-check').classList.add('opacity-0');
                });
                updateFilterCount();
                filterShops();
            });

            function updateCurrentTime() {
                const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                const now = new Date();
                currentDay = days[now.getDay()];

                let hours = now.getHours();
                let minutes = now.getMinutes();
                hours = hours < 10 ? '0' + hours : hours;
                minutes = minutes < 10 ? '0' + minutes : minutes;

                currentTime = `${hours}:${minutes}`;
                checkShopOpenStatus();
            }

            function timeToMinutes(timeString) {
                const [hours, minutes] = timeString.split(':').map(Number);
                return hours * 60 + minutes;
            }

            function isShopOpen(openDays, openHour, closeHour) {
                if (!openDays.includes(currentDay)) {
                    return false;
                }

                const currentTimeMinutes = timeToMinutes(currentTime);
                const openTimeMinutes = timeToMinutes(openHour);
                const closeTimeMinutes = timeToMinutes(closeHour);

                return currentTimeMinutes >= openTimeMinutes && currentTimeMinutes <= closeTimeMinutes;
            }

            function isClosingSoon(openDays, closeHour, category) {
                if (!openDays.includes(currentDay) || category !== 'Restaurant') {
                    return false;
                }

                const currentTimeMinutes = timeToMinutes(currentTime);
                const closeTimeMinutes = timeToMinutes(closeHour);
                const timeUntilClose = closeTimeMinutes - currentTimeMinutes;

                return timeUntilClose > 0 && timeUntilClose <= 120;
            }

            function calculateNextOpening(openDays, openHour) {
                if (openDays.length === 0) return null;

                const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                const currentDayIndex = days.indexOf(currentDay);
                const currentTimeMinutes = timeToMinutes(currentTime);
                const openTimeMinutes = timeToMinutes(openHour);

                if (openDays.includes(currentDay) && currentTimeMinutes < openTimeMinutes) {
                    return { day: currentDay, time: openHour };
                }

                for (let i = 1; i <= 7; i++) {
                    const nextDayIndex = (currentDayIndex + i) % 7;
                    const nextDay = days[nextDayIndex];
                    if (openDays.includes(nextDay)) {
                        return { day: nextDay, time: openHour };
                    }
                }
                return null;
            }

            function formatNextOpeningDay(day) {
                const dayTranslations = {
                    'Monday': 'Lundi', 'Tuesday': 'Mardi', 'Wednesday': 'Mercredi',
                    'Thursday': 'Jeudi', 'Friday': 'Vendredi', 'Saturday': 'Samedi', 'Sunday': 'Dimanche'
                };
                return day === currentDay ? "Aujourd'hui" : dayTranslations[day];
            }

            function checkShopOpenStatus() {
                shopCards.forEach(shop => {
                    const shopId = shop.getAttribute('data-shop-id');
                    const openHour = shop.getAttribute('data-open-hour');
                    const closeHour = shop.getAttribute('data-close-hour');
                    const openDays = JSON.parse(shop.getAttribute('data-open-days'));
                    const category = shop.getAttribute('data-category');

                    const isOpen = isShopOpen(openDays, openHour, closeHour);
                    const closingSoon = isClosingSoon(openDays, closeHour, category);
                    
                    shopStatus[shopId] = isOpen;
                    shopClosingSoon[shopId] = closingSoon;

                    const statusBadge = shop.querySelector('.status-badge');
                    const statusText = shop.querySelector('.status-text');
                    const closingIndicator = shop.querySelector('.closing-soon-indicator');

                    if (statusBadge && statusText) {
                        if (isOpen) {
                            statusBadge.classList.remove('bg-red-500');
                            statusBadge.classList.add('bg-green-500');
                            statusText.textContent = 'Ouvert';
                        } else {
                            statusBadge.classList.remove('bg-green-500');
                            statusBadge.classList.add('bg-red-500');
                            statusText.textContent = 'Fermé';

                            const nextOpen = calculateNextOpening(openDays, openHour);
                            if (nextOpen) {
                                nextOpenings[shopId] = nextOpen;
                                const nextOpeningElement = shop.querySelector('.next-opening');
                                if (nextOpeningElement) {
                                    const nextOpeningText = nextOpeningElement.querySelector('.next-opening-text');
                                    if (nextOpeningText) {
                                        nextOpeningText.textContent = formatNextOpeningDay(nextOpen.day) + ' à ' + nextOpen.time;
                                        nextOpeningElement.style.display = 'block';
                                    }
                                }
                            }
                        }
                    }

                    if (closingIndicator) {
                        if (closingSoon) {
                            closingIndicator.classList.remove('hidden');
                        } else {
                            closingIndicator.classList.add('hidden');
                        }
                    }
                });
            }

            function filterShops() {
                const searchTerm = searchInput.value.toLowerCase();
                
                shopCards.forEach(shop => {
                    const shopId = shop.getAttribute('data-shop-id');
                    const title = shop.getAttribute('data-title').toLowerCase();
                    const description = shop.getAttribute('data-description').toLowerCase();
                    const morning = shop.getAttribute('data-morning') === 'true';
                    const late = shop.getAttribute('data-late') === 'true';
                    const weekend = shop.getAttribute('data-weekend') === 'true';
                    const delivery = shop.getAttribute('data-delivery') === 'true';
                    const noReservation = shop.getAttribute('data-no-reservation') === 'true';
                    const isOpen = shopStatus[shopId] || false;
                    const isClosingSoon = shopClosingSoon[shopId] || false;

                    const matchesSearch = !searchTerm || title.includes(searchTerm) || description.includes(searchTerm);
                    const matchesFilters = (!activeFilters.has('open-now') || isOpen) &&
                                         (!activeFilters.has('last-minute') || isClosingSoon) &&
                                         (!activeFilters.has('no-reservation') || noReservation) &&
                                         (!activeFilters.has('morning') || morning) &&
                                         (!activeFilters.has('late') || late) &&
                                         (!activeFilters.has('weekend') || weekend) &&
                                         (!activeFilters.has('delivery') || delivery);

                    shop.style.display = matchesSearch && matchesFilters ? 'block' : 'none';
                });
            }

            searchInput.addEventListener('input', filterShops);

            document.querySelectorAll('.filter-button').forEach(button => {
                button.addEventListener('click', function() {
                    const filterId = this.id.replace('filter-', '');
                    const checkIcon = this.querySelector('.fa-check');
                    
                    if (activeFilters.has(filterId)) {
                        activeFilters.delete(filterId);
                        this.classList.remove('bg-black', 'dark:bg-white', 'text-white', 'dark:text-black');
                        this.classList.add('bg-gray-50', 'dark:bg-stone-900');
                        checkIcon.classList.add('opacity-0');
                    } else {
                        activeFilters.add(filterId);
                        this.classList.add('bg-black', 'dark:bg-white', 'text-white', 'dark:text-black');
                        this.classList.remove('bg-gray-50', 'dark:bg-stone-900');
                        checkIcon.classList.remove('opacity-0');
                    }
                    
                    updateFilterCount();
                    filterShops();
                });
            });

            document.querySelectorAll('.schedule-button').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    const shopId = this.getAttribute('data-shop-id');
                    const scheduleDetails = document.getElementById('schedule-' + shopId);
                    scheduleDetails.style.display = scheduleDetails.style.display === 'none' ? 'block' : 'none';
                });
            });

            document.querySelectorAll('[data-shop]').forEach(card => {
                card.addEventListener('click', function() {
                    const shopId = this.getAttribute('data-shop-id');
                    window.location.href = 'shop-details-mobile.php?id=' + shopId;
                });
            });

            updateCurrentTime();
            setInterval(updateCurrentTime, 60000);
        });
    </script>
</body>
</html>