<?php
session_start();
require_once __DIR__ . '/vendor/autoload.php';
require_once 'db.php';

$stripe_config = require 'config/stripe.php';
\Stripe\Stripe::setApiKey($stripe_config['secret_key']);

if (!isset($_SESSION['resident_id'])) {
    header('Location: login.php');
    exit;
}

$resident_id = $_SESSION['resident_id'];
$token = $_GET['token'] ?? '';
$user_id = $_GET['user_id'] ?? '';

if (empty($token) || $user_id != $resident_id) {
    die('Token invalide ou utilisateur non autorisé');
}

$stmt = $conn->prepare("SELECT * FROM residents WHERE id = ?");
$stmt->bind_param("i", $resident_id);
$stmt->execute();
$resident = $stmt->get_result()->fetch_assoc();

$existing_sub = $conn->prepare("SELECT * FROM subscriptions WHERE resident_id = ? AND status IN ('active', 'trialing')");
$existing_sub->bind_param("i", $resident_id);
$existing_sub->execute();
$subscription = $existing_sub->get_result()->fetch_assoc();

if ($subscription) {
    header('Location: dashboard.php');
    exit;
}

$error_message = '';
$success_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['payment_method_id'])) {
    try {
        $payment_method_id = $_POST['payment_method_id'];
        
        $customer = \Stripe\Customer::create([
            'payment_method' => $payment_method_id,
            'email' => $resident['email'],
            'name' => $resident['first_name'] . ' ' . $resident['last_name'],
            'metadata' => [
                'resident_id' => $resident_id
            ],
            'invoice_settings' => [
                'default_payment_method' => $payment_method_id,
            ],
        ]);

        $subscription = \Stripe\Subscription::create([
            'customer' => $customer->id,
            'items' => [['price' => $stripe_config['price_id']]],
            'expand' => ['latest_invoice.payment_intent'],
            'metadata' => [
                'resident_id' => $resident_id
            ]
        ]);

        $stmt = $conn->prepare("INSERT INTO subscriptions (resident_id, stripe_customer_id, stripe_subscription_id, stripe_price_id, status, current_period_start, current_period_end) VALUES (?, ?, ?, ?, ?, FROM_UNIXTIME(?), FROM_UNIXTIME(?))");
        $status = $subscription->status;
        $stmt->bind_param("issssii", 
            $resident_id, 
            $customer->id, 
            $subscription->id, 
            $stripe_config['price_id'], 
            $status, 
            $subscription->current_period_start, 
            $subscription->current_period_end
        );
        $stmt->execute();
        $subscription_db_id = $conn->insert_id;

        if ($subscription->status === 'active') {
            $update_member = $conn->prepare("UPDATE residents SET member_ = 1 WHERE id = ?");
            $update_member->bind_param("i", $resident_id);
            $update_member->execute();
        }

        $latest_invoice = $subscription->latest_invoice;
        if ($latest_invoice) {
            $payment_stmt = $conn->prepare("INSERT INTO payment_history (resident_id, subscription_id, stripe_payment_intent_id, amount, currency, status, payment_date) VALUES (?, ?, ?, ?, ?, ?, NOW())");
            $amount = $latest_invoice->amount_paid / 100;
            $currency = $latest_invoice->currency;
            $payment_status = $latest_invoice->status;
            $payment_intent_id = $latest_invoice->payment_intent;
            $payment_stmt->bind_param("iisdss", 
                $resident_id, 
                $subscription_db_id, 
                $payment_intent_id, 
                $amount, 
                $currency, 
                $payment_status
            );
            $payment_stmt->execute();
        }

        $success_message = 'Abonnement créé avec succès !';
        header('Location: dashboard.php?subscription_success=1');
        exit;

    } catch (\Stripe\Exception\CardException $e) {
        $error_message = 'Erreur de paiement : ' . $e->getError()->message;
    } catch (Exception $e) {
        $error_message = 'Erreur : ' . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Abonnement Premium - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://js.stripe.com/v3/"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
        }

        .StripeElement {
            background-color: white;
            padding: 16px;
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            transition: border-color 0.2s ease;
        }

        .dark .StripeElement {
            background-color: rgba(255, 255, 255, 0.1);
            border-color: rgba(255, 255, 255, 0.2);
        }

        .StripeElement--focus {
            border-color: #16a34a;
        }

        .StripeElement--invalid {
            border-color: #dc2626;
        }

        .loading-spinner {
            display: none;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-top: 3px solid white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            animation: spin 0.8s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .feature-item {
            display: flex;
            align-items: start;
            gap: 12px;
            padding: 16px;
            background: rgba(22, 163, 74, 0.05);
            border-radius: 12px;
            margin-bottom: 12px;
        }

        .dark .feature-item {
            background: rgba(22, 163, 74, 0.1);
        }
    </style>
</head>

<body class="bg-gray-50 dark:bg-gray-900 transition-colors duration-300">
    <script>
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        }
    </script>

    <div class="min-h-screen py-12 px-4">
        <div class="max-w-2xl mx-auto">
            <div class="text-center mb-8">
                <h1 class="text-3xl md:text-4xl font-light text-gray-900 dark:text-white mb-2">
                    Abonnement Premium
                </h1>
                <p class="text-gray-600 dark:text-gray-400">
                    Accède à tous les avantages exclusifs
                </p>
            </div>

            <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-lg p-6 md:p-8 mb-6">
                <div class="text-center mb-8">
                    <div class="inline-flex items-baseline gap-1 mb-2">
                        <span class="text-5xl font-bold text-gray-900 dark:text-white">1,99€</span>
                        <span class="text-gray-600 dark:text-gray-400">/mois</span>
                    </div>
                    <p class="text-sm text-gray-500 dark:text-gray-400">Annule à tout moment</p>
                </div>

                <div class="space-y-3 mb-8">
                    <div class="feature-item">
                        <svg class="w-6 h-6 text-green-600 flex-shrink-0 mt-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                        <div>
                            <h3 class="font-semibold text-gray-900 dark:text-white">Carte de membre digitale</h3>
                            <p class="text-sm text-gray-600 dark:text-gray-400">Accès à ta carte avec code-barres scannable</p>
                        </div>
                    </div>

                    <div class="feature-item">
                        <svg class="w-6 h-6 text-green-600 flex-shrink-0 mt-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                        <div>
                            <h3 class="font-semibold text-gray-900 dark:text-white">10% de réduction</h3>
                            <p class="text-sm text-gray-600 dark:text-gray-400">Chez tous les partenaires du quartier</p>
                        </div>
                    </div>

                    <div class="feature-item">
                        <svg class="w-6 h-6 text-green-600 flex-shrink-0 mt-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                        <div>
                            <h3 class="font-semibold text-gray-900 dark:text-white">Offres exclusives</h3>
                            <p class="text-sm text-gray-600 dark:text-gray-400">Accès prioritaire aux promotions spéciales</p>
                        </div>
                    </div>

                    <div class="feature-item">
                        <svg class="w-6 h-6 text-green-600 flex-shrink-0 mt-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                        <div>
                            <h3 class="font-semibold text-gray-900 dark:text-white">Support local</h3>
                            <p class="text-sm text-gray-600 dark:text-gray-400">Aide l'économie du quartier</p>
                        </div>
                    </div>
                </div>

                <?php if ($error_message): ?>
                <div class="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4 mb-6">
                    <p class="text-red-800 dark:text-red-200 text-sm"><?= htmlspecialchars($error_message) ?></p>
                </div>
                <?php endif; ?>

                <form id="payment-form" method="POST">
                    <div class="mb-6">
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Informations de paiement
                        </label>
                        <div id="card-element"></div>
                        <div id="card-errors" class="text-red-600 text-sm mt-2"></div>
                    </div>

                    <button type="submit" id="submit-button" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-4 rounded-xl transition-colors duration-200 flex items-center justify-center gap-2">
                        <span id="button-text">S'abonner maintenant</span>
                        <div class="loading-spinner" id="spinner"></div>
                    </button>

                    <p class="text-xs text-center text-gray-500 dark:text-gray-400 mt-4">
                        Paiement sécurisé par Stripe. Tu seras débité de 1,99€ par mois.
                    </p>
                </form>
            </div>

            <div class="text-center">
                <a href="resident-dashboard.php" class="text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white text-sm">
                    ← Retour au tableau de bord
                </a>
            </div>
        </div>
    </div>

    <script>
        const stripe = Stripe('<?= $stripe_config['publishable_key'] ?>');
        const elements = stripe.elements();

        const style = {
            base: {
                color: document.documentElement.classList.contains('dark') ? '#ffffff' : '#000000',
                fontFamily: 'Poppins, sans-serif',
                fontSize: '16px',
                '::placeholder': {
                    color: document.documentElement.classList.contains('dark') ? 'rgba(255, 255, 255, 0.5)' : 'rgba(0, 0, 0, 0.4)',
                }
            },
            invalid: {
                color: '#dc2626',
            }
        };

        const cardElement = elements.create('card', { style: style });
        cardElement.mount('#card-element');

        cardElement.on('change', function(event) {
            const displayError = document.getElementById('card-errors');
            if (event.error) {
                displayError.textContent = event.error.message;
            } else {
                displayError.textContent = '';
            }
        });

        const form = document.getElementById('payment-form');
        const submitButton = document.getElementById('submit-button');
        const buttonText = document.getElementById('button-text');
        const spinner = document.getElementById('spinner');

        form.addEventListener('submit', async function(event) {
            event.preventDefault();
            
            submitButton.disabled = true;
            buttonText.textContent = 'Traitement...';
            spinner.style.display = 'block';

            const {paymentMethod, error} = await stripe.createPaymentMethod({
                type: 'card',
                card: cardElement,
                billing_details: {
                    email: '<?= htmlspecialchars($resident['email']) ?>',
                    name: '<?= htmlspecialchars($resident['first_name'] . ' ' . $resident['last_name']) ?>'
                }
            });

            if (error) {
                document.getElementById('card-errors').textContent = error.message;
                submitButton.disabled = false;
                buttonText.textContent = 'S\'abonner maintenant';
                spinner.style.display = 'none';
            } else {
                const hiddenInput = document.createElement('input');
                hiddenInput.setAttribute('type', 'hidden');
                hiddenInput.setAttribute('name', 'payment_method_id');
                hiddenInput.setAttribute('value', paymentMethod.id);
                form.appendChild(hiddenInput);
                form.submit();
            }
        });
    </script>
</body>
</html>